/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include "lang.h"

#include "common.h"
#include "config_wrapper.h"
#include "xine-toolkit/menu.h"
#include "kbindings.h"
#include "menus.h"
#include "acontrol.h"
#include "control.h"
#include "event_sender.h"
#include "mrl_browser.h"
#include "panel.h"
#include "playlist.h"
#include "videowin.h"
#include "actions.h"
#include "event.h"
#include "oxine/oxine.h"

#define PLAYL_NO_LOOP    (PLAYLIST_LAST + 0)
#define PLAYL_LOOP       (PLAYLIST_LAST + 1)
#define PLAYL_REPEAT     (PLAYLIST_LAST + 2)
#define PLAYL_SHUFFLE    (PLAYLIST_LAST + 3)
#define PLAYL_SHUF_PLUS  (PLAYLIST_LAST + 4)
#define PLAYL_GET_FROM   (PLAYLIST_LAST + 5)
#define PLAYL_OPEN_MRLB  (PLAYLIST_LAST + 6)


#define AUDIO_DECRE_VOL          0
#define AUDIO_INCRE_VOL          1

#define CTRL_RESET               0

#define _MENU_GUI_ACTION_BASE   (0 << 24)
#define _MENU_ASPECT_BASE       (1 << 24)
#define _MENU_AUDIO_CMD_BASE    (2 << 24)
#define _MENU_AUDIO_CHAN_BASE   ((3 << 24) + 2)
#define _MENU_AUDIO_VIZ_BASE    (4 << 24)
#define _MENU_SUBT_CHAN_BASE    ((5 << 24) + 2)
#define _MENU_PLAYL_CMD_BASE    (6 << 24)
#define _MENU_CTRL_CMD_BASE     (7 << 24)
#define _MENU_VOL_BASE          (8 << 24)

typedef struct {
  char *write, *end, buf[1024];
} menu_text_buf_t;

static void _menu_set_shortcuts (gGui_t *gui, menu_text_buf_t *tbuf, xitk_menu_entry_t *m) {
  if (!(gui->kbindings_enabled & 1)) {
    for (; m->type != XITK_MENU_ENTRY_END; m++)
      m->shortcut = NULL;
    return;
  }
  for (; m->type != XITK_MENU_ENTRY_END; m++) {
    size_t sz;

    if (!m->shortcut)
      continue;
    if (tbuf->write + 1 >= tbuf->end) {
      if (gui->verbosity >= 1)
        printf ("gui.menu: text buffer overflow.\n");
      break;
    }
    sz = kbindings_shortcut_from_name (gui->kbindings, m->shortcut, tbuf->write, tbuf->end - tbuf->write, gui->shortcut_style);
    if (!sz) {
      /* no shortcut set (VOID) */
      m->shortcut = NULL;
      continue;
    }
    m->shortcut = tbuf->write;
    tbuf->write += sz + 1;
  }
  for (; m->type != XITK_MENU_ENTRY_END; m++)
    m->shortcut = NULL;
}

static void _menu_action (xitk_widget_t *w, xitk_menu_entry_t *me, void *data) {
  gGui_t *gui = data;

  (void)w;
  if (!me)
    return;

  switch (me->user_id >> 24) {

    case (_MENU_GUI_ACTION_BASE >> 24):
      gui_action_id (gui, me->user_id - _MENU_GUI_ACTION_BASE);
      break;

    case (_MENU_ASPECT_BASE >> 24):
      gui_toggle_aspect (gui, me->user_id - _MENU_ASPECT_BASE);
      break;

    case (_MENU_AUDIO_CMD_BASE >> 24):
      switch (me->user_id - _MENU_AUDIO_CMD_BASE) {
        static const uint8_t _step[LAST_MIXER] = {
          [SOUND_CARD_MIXER] = 100 / 10, [SOFTWARE_MIXER] = 200 / 10
        };
        case AUDIO_DECRE_VOL:
        case AUDIO_INCRE_VOL:
          gui_set_audio_vol (gui, GUI_AUDIO_VOL_RELATIVE +
            (int)_step[gui->mixer.type_volume] * (2 * ((int)me->user_id - _MENU_AUDIO_CMD_BASE - AUDIO_DECRE_VOL) - 1));
          break;
        default:
          printf("%s(): unknown control %d\n", __XINE_FUNCTION__, me->user_id - _MENU_CTRL_CMD_BASE);
          break;
      }
      break;

    case (_MENU_AUDIO_CHAN_BASE >> 24):
      gui_direct_change_audio_channel (w, gui, me->user_id - _MENU_AUDIO_CHAN_BASE);
      break;

    case (_MENU_AUDIO_VIZ_BASE >> 24):
      config_update_num (gui->xine, "gui.post_audio_plugin", me->user_id - _MENU_AUDIO_VIZ_BASE);
      break;

    case (_MENU_SUBT_CHAN_BASE >> 24):
      gui_direct_change_spu_channel (w, gui, me->user_id - _MENU_SUBT_CHAN_BASE);
      break;

    case (_MENU_PLAYL_CMD_BASE >> 24):
      switch (me->user_id - _MENU_PLAYL_CMD_BASE) {
        case PLAYLIST_LOAD:
        case PLAYLIST_SAVE:
        case PLAYLIST_CURR_REMOVE:
        case PLAYLIST_CLEAR:
        case PLAYLIST_CURR_UP:
        case PLAYLIST_CURR_DOWN:
	case PLAYLIST_CURR_PLAY:
        case PLAYLIST_CURR_SCAN:
	case PLAYLIST_SCAN:
          playlist_action (gui, me->user_id - _MENU_PLAYL_CMD_BASE);
          break;
        case PLAYL_NO_LOOP:
        case PLAYL_LOOP:
        case PLAYL_REPEAT:
        case PLAYL_SHUFFLE:
        case PLAYL_SHUF_PLUS:
          gui_action_args (gui, ACTID_LOOPMODE, me->user_id - _MENU_PLAYL_CMD_BASE - PLAYL_NO_LOOP, NULL);
          break;
        case PLAYL_GET_FROM:
          playlist_scan_input_s (gui, me->menu);
          break;
        case PLAYL_OPEN_MRLB:
          open_mrlbrowser_from_playlist (w, gui, 0, 0);
          break;
        default:
          printf ("%s(): unknown control %d\n", __XINE_FUNCTION__, me->user_id - _MENU_PLAYL_CMD_BASE);
      }
      break;

    case (_MENU_CTRL_CMD_BASE >> 24):
      switch (me->user_id - _MENU_CTRL_CMD_BASE) {
        case CTRL_RESET:
          control_reset (gui->vctrl);
          break;
        default: ;
      }
      break;

    case (_MENU_VOL_BASE >> 24):
      switch (me->user_id - _MENU_VOL_BASE) {
        case SOUND_CARD_MIXER:
        case SOFTWARE_MIXER:
          if ((int)gui->mixer.type_volume == me->user_id - _MENU_VOL_BASE)
            break;
          gui->mixer.type_volume = me->user_id - _MENU_VOL_BASE;
          panel_update_mixer_display (gui->panel);
          break;
        case LAST_MIXER:
          acontrol_main (XUI_W_ON, gui);
          break;
        default: ;
      }
      break;

    default: ;
  }
}

#define _menu_check_cond(_cond) (_cond ? XITK_MENU_ENTRY_CHECKED : XITK_MENU_ENTRY_CHECK)
#define _menu_check_val(_val,_test) ((((_val + 0x3fffffff - _test) ^ (_val + 0x40000000 - _test)) >> 30) + XITK_MENU_ENTRY_CHECK)
#define _menu_check_flag(_val,_flag) (xitk_bitmove (_val, _flag, 1) + XITK_MENU_ENTRY_CHECK)

#ifdef HAVE_PTR_TO_PROTECTED_VISIBILITY_FUNC
#  define _xine_get_audio_lang xine_get_audio_lang
#  define _xine_get_spu_lang xine_get_spu_lang
#else
static int _xine_get_audio_lang (xine_stream_t *stream, int channel, char *buf) {
  return xine_get_audio_lang (stream, channel, buf);
}
static int _xine_get_spu_lang (xine_stream_t *stream, int channel, char *buf) {
  return xine_get_spu_lang (stream, channel, buf);
}
#endif

static void _menu_add_channels (gGui_t *gui, xitk_widget_t *w, unsigned int mode) {
  static const char num15[] = " 0 1 2 3 4 5 6 7 8 91011121314";
  static const unsigned int xine_param[2] = { XINE_PARAM_AUDIO_CHANNEL_LOGICAL, XINE_PARAM_SPU_CHANNEL };
  typedef int (*get_func) (xine_stream_t *stream, int i, char *buf);
  static const get_func get_fn[2] = { _xine_get_audio_lang, _xine_get_spu_lang };
  static const unsigned int base[2] = { _MENU_AUDIO_CHAN_BASE, _MENU_SUBT_CHAN_BASE };
  xitk_menu_entry_t menu_entry;
  int channel = xine_get_param (gui->stream, xine_param[mode]);
  uint8_t type[34];
  char buf[XINE_LANG_MAX];
  int i;

  memset (type, XITK_MENU_ENTRY_CHECK, sizeof (type));
  if (XITK_0_TO_MAX_MINUS_1 (channel + 2, 34))
    type[channel + 2] = XITK_MENU_ENTRY_CHECKED;
  menu_entry.shortcut = NULL;
  menu_entry.user_id = base[mode] - 2;

  menu_entry.type = type[0];
  menu_entry.menu = _("Off");
  xitk_menu_add_entry (w, &menu_entry);
  menu_entry.user_id++;
  menu_entry.type = type[1];
  menu_entry.menu = _("Auto");
  xitk_menu_add_entry (w, &menu_entry);
  menu_entry.user_id++;
  menu_entry.type = XITK_MENU_ENTRY_SEPARATOR;
  menu_entry.menu = "SEP";
  xitk_menu_add_entry (w, &menu_entry);

  for (i = 0; i < 32; i++) {
    buf[0] = 0;
    if (!get_fn[mode] (gui->stream, i, buf))
      break;
    menu_entry.type = type[i + 2];
    menu_entry.menu = get_language_from_iso639_1 (buf);
    xitk_menu_add_entry (w, &menu_entry);
    menu_entry.user_id++;
  }
  if (i == 0) {
    buf[2] = 0;
    menu_entry.menu = buf;
    for (i = 0; i < 15; i++) {
      memcpy (buf, num15 + 2 * i, 2);
      menu_entry.type = type[i + 2];
      xitk_menu_add_entry (w, &menu_entry);
      menu_entry.user_id++;
    }
  }
}

/* fake string */
#define _FS(_s) (const char *)(uintptr_t)(_s)

void video_window_menu (gGui_t *gui, xitk_widget_list_t *wl, int x, int y) {
  unsigned int         aspect = xine_get_param (gui->stream, XINE_PARAM_VO_ASPECT_RATIO) & 0x3fffffff;
/* cd po && make update-po
 * will copy the next comment to the *.po files. */
/* TRANSLATORS: the next snippets define a menu tree.
 * Wherever you got something like "foo", "foo/bar" and "foo/bar/buff",
 * make sure to translate all "foo"s the same way,
 * and all "bar"s another same way.
 * Also, just leave "SEP" as is.
 * Different translations for the same word sometimes are good style.
 * However, here they will create unintended extra branches. */
  const char          *play_pause = _("Playback/Pause");
  const char          *playlist_from = _("Playlist/Get from");
  const char          *menus_nav = _("Menus/Navigation...");
  const char          *video_deint = _("Video/Deinterlace");
  const char          *audio_vis = _("Audio/Visualization");
  const char          *audio_chan = _("Audio/Channel");
  const char          *subt_chan = _("Subtitle/Channel");
  const char          *net_remote = _("Settings/Network remote control");
  unsigned int         loop_mode = gui->playlist.loop;
  char                 title[256];
  xitk_widget_t       *w;
  vwin_mag_t           mag = video_window_set_mag (gui->vwin, -VWIN_MAG_1);
  int                  vwin_mode = video_window_mode (gui->vwin, TOGGLE_MODE);
  menu_text_buf_t      tbuf = {tbuf.buf, tbuf.buf + sizeof (tbuf.buf) - 1, {0}};
  const char          *menu_names[7];
  int                  num_menus = event_sender_menu_names (gui, menu_names);
  xitk_menu_entry_t    menu_entries[] = {
    { XITK_MENU_ENTRY_TITLE, 0, title, NULL},
    { _menu_check_flag (panel_is_visible (gui->panel), 2), _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_VISIBLITY,
      _("Show controls"), _FS (ACTID_TOGGLE_VISIBLITY)},
    { _menu_check_flag (video_window_is_visible (gui->vwin), 2), _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_WINOUT_VISIBLITY,
      _("Show video window"), _FS (ACTID_TOGGLE_WINOUT_VISIBLITY)},
    { _menu_check_flag (vwin_mode, FULLSCR_MODE), _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_FULLSCREEN,
      _("Fullscreen"), _FS (ACTID_TOGGLE_FULLSCREEN)},
#ifdef HAVE_XINERAMA
#  define _MENU_OFFS1 1
    { _menu_check_flag (vwin_mode, FULLSCR_XI_MODE), _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_XINERAMA_FULLSCREEN,
      _("Xinerama fullscreen"), _FS (ACTID_TOGGLE_XINERAMA_FULLSCREEN)},
#else
#  define _MENU_OFFS1 0
#endif
    { _menu_check_flag (vwin_mode, BORDER_MODE), _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_WINOUT_BORDER,
      _("Window frame"), _FS (ACTID_TOGGLE_WINOUT_BORDER)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL},
    /* "Open" */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_FILESELECTOR,
      _("Open/File..."), _FS (ACTID_FILESELECTOR)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_PLAYL_CMD_BASE + PLAYLIST_LOAD,
      _("Open/Playlist..."), _FS (ACTID_PLAYLIST_OPEN)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_MRLBROWSER,
      _("Open/Location..."), _FS (ACTID_MRLBROWSER)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SUBSELECT,
      _("Open/Subtitle..."), _FS (ACTID_SUBSELECT)},
    /* "Playback" */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_PLAY, _("Playback/Play"), _FS (ACTID_PLAY)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_STOP, _("Playback/Stop"), _FS (ACTID_STOP)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_PAUSE, play_pause, _FS (ACTID_PAUSE)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, play_pause, NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_MRL_NEXT, _("Playback/Next MRL"), _FS (ACTID_MRL_NEXT)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_MRL_PRIOR, _("Playback/Previous MRL"), _FS (ACTID_MRL_PRIOR)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, play_pause, NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SPEED_FAST,
      _("Playback/Increase Speed"), _FS (ACTID_SPEED_FAST)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SPEED_SLOW,
      _("Playback/Decrease Speed"), _FS (ACTID_SPEED_SLOW)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, play_pause, NULL},
    { _menu_check_cond (gui->is_display_mrl), _MENU_GUI_ACTION_BASE + ACTID_SHOW_MRL,
      _("Playback/Show MRL"), _FS (ACTID_SHOW_MRL)},
    { _menu_check_cond (gui->runtime_mode), _MENU_GUI_ACTION_BASE + ACTID_SHOW_TIME,
      _("Playback/Show remaining time"), _FS (ACTID_SHOW_TIME)},
    /* "Playlist" */
    { XITK_MENU_ENTRY_BRANCH, 0, playlist_from, NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_PLAYL_CMD_BASE + PLAYLIST_LOAD, _("Playlist/Load..."), _FS (ACTID_PLAYLIST_OPEN)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_PLAYLIST,
      _("Playlist/Editor..."), _FS (ACTID_PLAYLIST)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, playlist_from, NULL},
    { XITK_MENU_ENTRY_BRANCH, 0, _("Playlist/Loop modes"), _FS (ACTID_LOOPMODE)},
    { _menu_check_val (loop_mode, PLAYLIST_LOOP_NO_LOOP), _MENU_PLAYL_CMD_BASE + PLAYL_NO_LOOP,
      _("Playlist/Loop modes/Disabled"), NULL},
    { _menu_check_val (loop_mode, PLAYLIST_LOOP_LOOP), _MENU_PLAYL_CMD_BASE + PLAYL_LOOP,
      _("Playlist/Loop modes/Loop"), NULL},
    { _menu_check_val (loop_mode, PLAYLIST_LOOP_REPEAT), _MENU_PLAYL_CMD_BASE + PLAYL_REPEAT,
      _("Playlist/Loop modes/Repeat Selection"), NULL},
    { _menu_check_val (loop_mode, PLAYLIST_LOOP_SHUFFLE), _MENU_PLAYL_CMD_BASE + PLAYL_SHUFFLE,
      _("Playlist/Loop modes/Shuffle"), NULL},
    { _menu_check_val (loop_mode, PLAYLIST_LOOP_SHUF_PLUS), _MENU_PLAYL_CMD_BASE + PLAYL_SHUF_PLUS,
      _("Playlist/Loop modes/Non-stop Shuffle"), NULL},
    { _menu_check_flag (~gui->playlist.control, PLAYLIST_CONTROL_STOP), _MENU_GUI_ACTION_BASE + ACTID_PLAYLIST_STOP,
      _("Playlist/Continue Playback"), _FS (ACTID_PLAYLIST_STOP)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_PLAYL_CMD_BASE + PLAYLIST_SAVE,
      _("Playlist/Save..."), NULL},
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL},
    /* "Menus" */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_SENDER, menus_nav, _FS (ACTID_EVENT_SENDER)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, menus_nav, NULL},
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL},
    /* "Stream" */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_STREAM_INFOS,
      _("Stream/Information..."), _FS (ACTID_STREAM_INFOS)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_OSD_SINFOS,
      _("Stream/Information (OSD)"), _FS (ACTID_OSD_SINFOS)},
    /* "Video" */
    { _menu_check_flag (gui->flags, XUI_FLAG_deint), _MENU_GUI_ACTION_BASE + ACTID_TOGGLE_INTERLEAVE,
      video_deint, _FS (ACTID_TOGGLE_INTERLEAVE)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, video_deint, NULL},
    { XITK_MENU_ENTRY_BRANCH, 0,
      _("Video/Aspect ratio"), _FS (ACTID_TOGGLE_ASPECT_RATIO)},
    { _menu_check_val (aspect, XINE_VO_ASPECT_AUTO), _MENU_ASPECT_BASE + XINE_VO_ASPECT_AUTO,
      _("Video/Aspect ratio/Automatic"), NULL},
    { _menu_check_val (aspect, XINE_VO_ASPECT_SQUARE), _MENU_ASPECT_BASE + XINE_VO_ASPECT_SQUARE,
      _("Video/Aspect ratio/Square"), NULL},
    { _menu_check_val (aspect, XINE_VO_ASPECT_4_3), _MENU_ASPECT_BASE + XINE_VO_ASPECT_4_3,
      _("Video/Aspect ratio/4:3"), NULL},
    { _menu_check_val (aspect, XINE_VO_ASPECT_ANAMORPHIC), _MENU_ASPECT_BASE + XINE_VO_ASPECT_ANAMORPHIC,
      _("Video/Aspect ratio/Anamorphic"), NULL},
    { _menu_check_val (aspect, XINE_VO_ASPECT_DVB), _MENU_ASPECT_BASE + XINE_VO_ASPECT_DVB,
      _("Video/Aspect ratio/DVB"), NULL},
    { _menu_check_cond (mag == 2 * VWIN_MAG_1), _MENU_GUI_ACTION_BASE + ACTID_WINDOW200,
      _("Video/200%"), _FS (ACTID_WINDOW200)},
    { _menu_check_cond (mag == VWIN_MAG_1), _MENU_GUI_ACTION_BASE + ACTID_WINDOW100,
      _("Video/100%"), _FS (ACTID_WINDOW100)},
    { _menu_check_cond (mag == VWIN_MAG_1 / 2), _MENU_GUI_ACTION_BASE + ACTID_WINDOW50,
      _("Video/50%"), _FS (ACTID_WINDOW50)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_WINDOWENLARGE,
      _("Video/larger"), _FS (ACTID_WINDOWENLARGE)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_WINDOWREDUCE,
      _("Video/smaller"), _FS (ACTID_WINDOWREDUCE)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, video_deint, NULL},
    { _menu_check_flag (gui->transform.flags, 1), _MENU_GUI_ACTION_BASE + ACTID_FLIP_H,
      _("Video/mirrored"), _FS (ACTID_FLIP_H)},
    { _menu_check_flag (gui->transform.flags, 2), _MENU_GUI_ACTION_BASE + ACTID_FLIP_V,
      _("Video/upside down"), _FS (ACTID_FLIP_V)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, video_deint, NULL},
    /* "Video/Postprocess" */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_VPP,
      _("Video/Postprocess/Chain Reaction..."), _FS (ACTID_VPP)},
    { _menu_check_flag (gui->flags, XUI_FLAG_post_v), _MENU_GUI_ACTION_BASE + ACTID_VPP_ENABLE,
      _("Video/Postprocess/Enable Postprocessing"), _FS (ACTID_VPP_ENABLE)},
    /* "Audio" */
    /* "Audio/Volume" */
    { _menu_check_cond (gui->mixer.mute[gui->mixer.type_mute]), _MENU_GUI_ACTION_BASE + ACTID_MUTE,
      _("Audio/Volume/Mute"), _FS (ACTID_MUTE)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_AUDIO_CMD_BASE + AUDIO_INCRE_VOL,
      _("Audio/Volume/Increase 10%"), _FS (ACTID_pVOLUME)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_AUDIO_CMD_BASE + AUDIO_DECRE_VOL,
      _("Audio/Volume/Decrease 10%"), _FS (ACTID_mVOLUME)},
    { XITK_MENU_ENTRY_BRANCH, 0, audio_chan, NULL},
    { XITK_MENU_ENTRY_BRANCH, 0, audio_vis, NULL},
    { XITK_MENU_ENTRY_SEPARATOR, 0, audio_vis, NULL},
    /* "Audio/Postprocess" */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_APP,
      _("Audio/Postprocess/Chain Reaction..."), _FS (ACTID_APP)},
    { _menu_check_flag (gui->flags, XUI_FLAG_post_a), _MENU_GUI_ACTION_BASE + ACTID_APP_ENABLE,
      _("Audio/Postprocess/Enable Postprocessing"), _FS (ACTID_APP_ENABLE)},
    /* "Subtitle" */
    { XITK_MENU_ENTRY_BRANCH, 0, subt_chan, NULL},
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL},
    /* "Settings" */
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SETUP,
      _("Settings/Setup..."), _FS (ACTID_SETUP)},
#ifdef HAVE_CURL
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_SKINDOWNLOAD,
      _("Settings/Skin Downloader..."), _FS (ACTID_SKINDOWNLOAD)},
#endif
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_KBEDIT,
      _("Settings/Keymap Editor..."), _FS (ACTID_KBEDIT)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_CONTROLSHOW,
      _("Settings/Video..."), _FS (ACTID_CONTROLSHOW)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_ACONTROLSHOW,
      _("Settings/Audio..."), _FS (ACTID_ACONTROLSHOW)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_TVANALOG,
      _("Settings/TV Analog..."), _FS (ACTID_TVANALOG)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, net_remote, NULL},
    { _menu_check_cond (gui->network), _MENU_GUI_ACTION_BASE + ACTID_NET_REMOTE, net_remote, _FS (ACTID_NET_REMOTE)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_HELP_SHOW, _("Help..."), _FS (ACTID_HELP_SHOW)},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_VIEWLOG, _("Logs..."), _FS (ACTID_VIEWLOG)},
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL},
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_QUIT, _("Quit"), _FS (ACTID_QUIT)},
    /* Menus acess */
    { XITK_MENU_ENTRY_BASE, 0, _("Menus"), NULL },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_MENU1 + 0, menu_names[0], _FS (ACTID_EVENT_MENU1 + 0) },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_MENU1 + 1, menu_names[1], _FS (ACTID_EVENT_MENU1 + 1) },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_MENU1 + 2, menu_names[2], _FS (ACTID_EVENT_MENU1 + 2) },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_MENU1 + 3, menu_names[3], _FS (ACTID_EVENT_MENU1 + 3) },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_MENU1 + 4, menu_names[4], _FS (ACTID_EVENT_MENU1 + 4) },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_MENU1 + 5, menu_names[5], _FS (ACTID_EVENT_MENU1 + 5) },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_EVENT_MENU1 + 6, menu_names[6], _FS (ACTID_EVENT_MENU1 + 6) },
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL },
    { _menu_check_flag (oxine_is_visible (gui->oxine), 1), _MENU_GUI_ACTION_BASE + ACTID_OSD_MENU, _("OSD"), _FS (ACTID_OSD_MENU) },
    /* Mediamark */
    { XITK_MENU_ENTRY_BASE, 0, _("Playback"), NULL },
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_ADDMEDIAMARK, _("Add Mediamark"),  _FS (ACTID_ADDMEDIAMARK) },
    { XITK_MENU_ENTRY_PLAIN, _MENU_GUI_ACTION_BASE + ACTID_MMKEDITOR,    _("Edit Mediamark"), _FS (ACTID_MMKEDITOR) },
    { XITK_MENU_ENTRY_END, 0, NULL, NULL}
  };
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .menu_tree = menu_entries,
    .cb = _menu_action
  };
  xitk_menu_entry_t menu_entry[1] = {{ .shortcut = NULL }};

  (void)num_menus;
  if (gui->flags & XUI_FLAG_no_gui)
    return;

  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);

  snprintf (title, sizeof (title), _("xine %s"), VERSION);
  /* Subtitle loader */
  if (gui->playlist.num <= 0) {
    /* "Open/Subtitle..." */
    menu_entries[_MENU_OFFS1 + 9].type = XITK_MENU_ENTRY_SKIP;
    /* "Playlist/Save..." */
    menu_entries[_MENU_OFFS1 + 33].type = XITK_MENU_ENTRY_SKIP;
  }
  if (xine_get_status (gui->stream) == XINE_STATUS_STOP)
    /* "Playback/{Add/Edit} Mediamark" */
    menu_entries[sizeof (menu_entries) / sizeof (menu_entries[0]) - 5].type = XITK_MENU_ENTRY_END;
  _menu_set_shortcuts (gui, &tbuf, menu_entries);

  gui->nongui_error_msg = NULL;

  w = xitk_noskin_menu_create (&menu, x, y);

  { /* Autoplay plugins */
    const char * const *plug_ids = xine_get_autoplay_input_plugin_ids (gui->xine);

    menu_entry[0].type = XITK_MENU_ENTRY_BASE;
    menu_entry[0].menu = playlist_from;
    xitk_menu_add_entry (w, menu_entry);
    menu_entry[0].type = XITK_MENU_ENTRY_PLAIN;
    menu_entry[0].user_id = _MENU_PLAYL_CMD_BASE + PLAYL_GET_FROM;
    while (*plug_ids) {
      menu_entry[0].menu = *plug_ids++;
      xitk_menu_add_entry (w, menu_entry);
    }
  }

  { /* Audio Viz */
    const char * const *viz_names = post_get_audio_plugins_names (gui);

    menu_entry[0].type = XITK_MENU_ENTRY_BASE;
    if (viz_names && *viz_names) {
      menu_entry[0].menu = audio_vis;
      xitk_menu_add_entry (w, menu_entry);
      menu_entry[0].user_id = _MENU_AUDIO_VIZ_BASE;
      while (*viz_names) {
        menu_entry[0].menu = *viz_names++;
        menu_entry[0].type = _menu_check_val ((menu_entry[0].user_id & 0xffff), gui->visual_anim.post_plugin_num);
        xitk_menu_add_entry (w, menu_entry);
        menu_entry[0].user_id++;
      }
    } else {
      menu_entry[0].menu = NULL;
      xitk_menu_add_entry (w, menu_entry);
      menu_entry[0].menu = _("Audio/Visualization/None");
      menu_entry[0].type = XITK_MENU_ENTRY_CHECKED;
      menu_entry[0].user_id = 0;
      xitk_menu_add_entry (w, menu_entry);
    }
  }

  /* Audio channels */
  menu_entry[0].type = XITK_MENU_ENTRY_BASE;
  menu_entry[0].menu = audio_chan;
  xitk_menu_add_entry (w, menu_entry);
  _menu_add_channels (gui, w, 0);
  /* SPU channels */
  menu_entry[0].menu = subt_chan;
  xitk_menu_add_entry (w, menu_entry);
  _menu_add_channels (gui, w, 1);

  xitk_menu_show_menu (w);
}

void lang_menu (gGui_t *gui, xitk_widget_list_t *wl, int mode, int x, int y) {
  static const char title[2][10] = { N_("Audio"), N_("Subtitle") };
  xitk_menu_entry_t menu_entries[] = {
    { XITK_MENU_ENTRY_TITLE, 0, gettext (title[mode & 1]), NULL },
    { XITK_MENU_ENTRY_END, 0, NULL, NULL }
  };
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action,
    .menu_tree = menu_entries
  };
  xitk_widget_t *w;

  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);
  w = xitk_noskin_menu_create (&menu, x, y);
  _menu_add_channels (gui, w, mode & 1);
  xitk_menu_show_menu (w);
}

void playlist_menu (gGui_t *gui, xitk_widget_list_t *wl, int x, int y, uint32_t flags) {
  xitk_widget_t *w = NULL;
  menu_text_buf_t tbuf = {tbuf.buf, tbuf.buf + sizeof (tbuf.buf) - 1, {0}};
  xitk_menu_entry_t menu_entries[15], *e = menu_entries;
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action,
    .menu_tree = menu_entries
  };

  e->type = XITK_MENU_ENTRY_TITLE;
  e->user_id = 0;
  e->menu = _("Playlist");
  e->shortcut = NULL;
  e++;
  if (flags & PLAYLIST_MENU_FLAG_PLAY) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_PLAY;
    e->menu = _("Play");
    e->shortcut = NULL;
    e++;
  }
  e->type = _menu_check_cond (gui->is_display_mrl);
  e->user_id = _MENU_GUI_ACTION_BASE + ACTID_SHOW_MRL;
  e->menu = _("Show MRL");
  e->shortcut = _FS (ACTID_SHOW_MRL);
  e++;
  e->type = XITK_MENU_ENTRY_SEPARATOR;
  e->user_id = 0;
  e->menu = "SEP";
  e->shortcut = NULL;
  e++;
  e->type = XITK_MENU_ENTRY_PLAIN;
  e->user_id = (flags & PLAYLIST_MENU_FLAG_PLAY)
             ? (_MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_SCAN)
             : (_MENU_GUI_ACTION_BASE + ACTID_SCANPLAYLIST);
  e->menu = _("Scan");
  e->shortcut = _FS (ACTID_SCANPLAYLIST);
  e++;
  e->type = XITK_MENU_ENTRY_PLAIN;
  e->user_id = _MENU_PLAYL_CMD_BASE + PLAYL_OPEN_MRLB;
  e->menu = _("Add");
  e->shortcut = NULL;
  e++;
  if (flags & PLAYLIST_MENU_FLAG_PLAY) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_GUI_ACTION_BASE + ACTID_MMKEDITOR;
    e->menu = _("Edit");
    e->shortcut = _FS (ACTID_MMKEDITOR);
    e++;
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_REMOVE;
    e->menu = _("Delete");
    e->shortcut = NULL;
    e++;
  }
  if (flags & PLAYLIST_MENU_FLAG_CLEAR) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CLEAR;
    e->menu = _("Delete All");
    e->shortcut = NULL;
    e++;
  }
  if (flags & (PLAYLIST_MENU_FLAG_UP | PLAYLIST_MENU_FLAG_DOWN)) {
    e->type = XITK_MENU_ENTRY_SEPARATOR;
    e->user_id = 0;
    e->menu = "SEP";
    e->shortcut = NULL;
    e++;
  }
  if (flags & PLAYLIST_MENU_FLAG_UP) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_UP;
    e->menu = _("Move Up");
    e->shortcut = NULL;
    e++;
  }
  if (flags & PLAYLIST_MENU_FLAG_DOWN) {
    e->type = XITK_MENU_ENTRY_PLAIN;
    e->user_id = _MENU_PLAYL_CMD_BASE + PLAYLIST_CURR_DOWN;
    e->menu = _("Move Down");
    e->shortcut = NULL;
    e++;
  }
  e->type = XITK_MENU_ENTRY_END;
  e->user_id = 0;
  e->menu = NULL;
  e->shortcut = NULL;

  _menu_set_shortcuts (gui, &tbuf, menu_entries);
  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);
  w = xitk_noskin_menu_create (&menu, x, y);
  xitk_menu_show_menu (w);
}

void control_menu (gGui_t *gui, xitk_widget_list_t *wl, int x, int y) {
  xitk_menu_entry_t    menu_entries[] = {
    { XITK_MENU_ENTRY_TITLE, 0, _("Video Control"), NULL },
    { XITK_MENU_ENTRY_PLAIN, _MENU_CTRL_CMD_BASE + CTRL_RESET, _("Reset video settings"), "[BackSpace]" },
    { XITK_MENU_ENTRY_END,   0, NULL, NULL }
  };
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action,
    .menu_tree = menu_entries
  };
  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);
  xitk_widget_t *w = xitk_noskin_menu_create (&menu, x, y);

  xitk_menu_show_menu (w);
}

void volume_menu (gGui_t *gui, xitk_widget_list_t *wl, int x, int y) {
  char buf[128];
  xitk_menu_entry_t    menu_entries[] = {
    { XITK_MENU_ENTRY_TITLE, 0, _("Volume control"), NULL },
    { _menu_check_val (gui->mixer.type_volume, SOUND_CARD_MIXER),
      _MENU_VOL_BASE + SOUND_CARD_MIXER, _("Sound card"), NULL },
    { _menu_check_val (gui->mixer.type_volume, SOFTWARE_MIXER),
      _MENU_VOL_BASE + SOFTWARE_MIXER, _("Software"), NULL },
    { XITK_MENU_ENTRY_SEPARATOR, 0, "SEP", NULL },
/* TRANSLATORS: last of menu tree snippets. */
    { XITK_MENU_ENTRY_PLAIN, _MENU_VOL_BASE + LAST_MIXER, _("Settings..."), NULL },
    { XITK_MENU_ENTRY_END,   0, NULL, NULL }
  };
  xitk_menu_widget_t menu = {
    .nw = { .wl = wl, .userdata = gui },
    .cb = _menu_action,
    .menu_tree = menu_entries
  };
  xitk_widget_t *w;
  /* reduce server resource consumnption. */
  xitk_menu_hide_menu (gui->xitk);
  buf[0] = 0;
  if ((gui->kbindings_enabled & 1) &&
    (kbindings_shortcut_from_name (gui->kbindings, "AControlShow", buf, sizeof (buf), gui->shortcut_style) > 0))
    menu_entries[4].shortcut = buf;
  w = xitk_noskin_menu_create (&menu, x, y);
  xitk_menu_show_menu (w);
}

