/* sysalias.c -- Systems Aliases tree management.
 *
 * This file is part of TUA.
 * 
 *   Copyright (C) 1991,96  Lele Gaifax
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the license, or (at
 *   your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include    "tua.h"

typedef struct system_aliases
{
  char *System;
  char *Alias;
} system_aliases_t;

static BTREE Root = NULL;

static int
DEFUN (alias_compare, (sys1, sys2),
       CONST system_aliases_t * sys1 AND CONST system_aliases_t * sys2)
{
  int ret = strcmp (sys1->System, sys2->System);

  return ret;
}

static PTR
DEFUN (alias_new, (data),
       CONST PTR * data)
{
  system_aliases_t * sa;
  
  sa = (system_aliases_t *) xmalloc (sizeof (system_aliases_t));
  sa->System = savestring (((CONST system_aliases_t *) data)->System);
  sa->Alias = NULL;

  return (PTR) sa;
}
       
CONST char *
DEFUN (insert_system_alias, (sys, alias),
       CONST char *sys AND
       CONST char *alias)
{
  system_aliases_t * sr_new, sr;
  
  if (Root == NULL)
    Root = btree_new ((compare_func_t) alias_compare, (makenew_func_t) alias_new);

  sr.System = (char *) sys;
  sr_new = (system_aliases_t *) btree_insert (Root, &sr);
  if (! sr_new->Alias)
    sr_new->Alias = savestring (alias);
  
  return sr_new->Alias;
}

CONST char *
DEFUN (get_possible_system_alias, (sys),
       CONST char *sys)
{
  system_aliases_t * sa, sa_key;

  if (Root == NULL)
    return sys;
  
  sa_key.System = (char *) sys;
  sa = (system_aliases_t *) btree_search (Root, &sa_key);

#if USE_GLOB_ALIASES
  if (sa == NULL)
    return get_possible_system_glob_alias (sys);
#endif   

  if (sa == NULL)
    return sys;
  else
    return sa->Alias;
}

#if USE_GLOB_ALIASES

#include "globalias.h"

static glob_aliases_queue_t *SystemGlobQueue = QNULL;

void
DEFUN (insert_system_glob_alias, (sys, alias),
       CONST char *sys AND
       CONST char *alias)
{
  globalias_insert (&SystemGlobQueue, sys, alias);
}

CONST char *
DEFUN (get_possible_system_glob_alias, (sys),
       CONST char *sys)
{
  return globalias_search (SystemGlobQueue, sys);
}

#endif /* if USE_GLOB_ALIASES */

#if USE_TCL

int
DEFUN (tcl_insert_system_alias, (clientData, interp, argc, argv),
       ClientData clientData AND
       Tcl_Interp * interp AND
       int argc AND
       char *argv[])

{
  if (argc != 3)
    {
      sprintf (interp->result, _("Wrong number of Parameters: %s needs an SystemName and its Alias"), argv[0]);
      return TCL_ERROR;
    }
  insert_system_alias (argv[1], argv[2]);
  return TCL_OK;
}

#if USE_GLOB_ALIASES

int
DEFUN (tcl_insert_system_glob_alias, (clientData, interp, argc, argv),
       ClientData clientData AND
       Tcl_Interp * interp AND
       int argc AND
       char *argv[])

{
  if (argc != 3)
    {
      sprintf (interp->result, _("Wrong number of Parameters: %s needs an SystemName and its GlobAlias"), argv[0]);
      return TCL_ERROR;
    }
  insert_system_glob_alias (argv[1], argv[2]);
  return TCL_OK;
}

#endif /* if USE_GLOB_ALIASES */

#endif /* if USE_TCL */
