/*
*  Rubrica
*  file: statusbar.h
*
*  
*  Copyright (C) 2000-2005 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>


#include "statusbar.h"
#include "types.h"


enum {
  PROP_0,
  STATUSBAR_MESSAGE,
  STATUSBAR_EXTRA,
  STATUSBAR_IMAGES,
  STATUSBAR_RATE,
  STATUSBAR_LOCK
};


struct _RubricaStatusbarPrivate {
  gint     ctxid;
  gchar*   message;
  gboolean enable_extra;
  gboolean enable_images;
  RubricaPixmap rate; 
  RubricaPixmap lock;

  GtkWidget* extra;         // extra box
  GtkWidget* images;        // images's box
  GtkWidget* label;         // 
  GtkWidget* sep;
  GtkWidget* rate_img;      // rate image
  GtkWidget* lock_img;      // lock image

  gboolean dispose_has_run;
};

#define RUBRICA_STATUSBAR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o), \
                                          RUBRICA_STATUSBAR_TYPE,          \
					  RubricaStatusbarPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_statusbar_class_init   (RubricaStatusbarClass* klass);
static void rubrica_statusbar_init         (RubricaStatusbar* obj);

static void rubrica_statusbar_finalize     (RubricaStatusbar* self);
static void rubrica_statusbar_dispose      (RubricaStatusbar* self);


static void rubrica_statusbar_set_property (GObject* obj, guint property_id,
					    GValue* value,  GParamSpec* spec);
static void rubrica_statusbar_get_property (GObject* obj, guint property_id,
					    GValue* value, GParamSpec* spec);

static GdkPixbuf* load_pixbuf(gint pixmap);


static GdkPixbuf* 
load_pixbuf(gint pixmap)
{
  GtkIconTheme* theme;
  GdkPixbuf* pixbuf = NULL;   
  
  theme = gtk_icon_theme_get_default(); 
  
  switch(pixmap)
    {
    case PIXMAP_RATE_NONE:
      pixbuf = gtk_icon_theme_load_icon (theme, "rate1", 20, 0, NULL);
      break;
      
    case PIXMAP_RATE_FEW:
      pixbuf = gtk_icon_theme_load_icon (theme, "rate2", 20, 0, NULL);
      break;
      
    case PIXMAP_RATE_NORMAL:
      pixbuf = gtk_icon_theme_load_icon (theme, "rate3", 20, 0, NULL);
      break;
      
    case PIXMAP_RATE_HIGHT:
      pixbuf = gtk_icon_theme_load_icon (theme, "rate4", 20, 0, NULL);
      break;
      
    case PIXMAP_RATE_VERY_HIGHT:
      pixbuf = gtk_icon_theme_load_icon (theme, "rate5", 20, 0, NULL);
      break;
      
    case PIXMAP_LOCK:
      pixbuf = gtk_icon_theme_load_icon (theme, "stock_lock-ok", 18, 0, NULL);
      break;

    case PIXMAP_UNLOCK:
      pixbuf = gtk_icon_theme_load_icon (theme, "stock_lock-open", 
					 18, 0, NULL);
      break;
      
      
    default:
      break;
    }
  
  return pixbuf;
}


GType
rubrica_statusbar_get_type()
{
  static GType rubrica_bar_type = 0;
  
  if (!rubrica_bar_type)
    {
      static const GTypeInfo rubrica_bar_info =
	{
	  sizeof(RubricaStatusbarClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_statusbar_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaStatusbar),
	  0,
	  (GInstanceInitFunc) rubrica_statusbar_init
	};

      rubrica_bar_type = g_type_register_static (GTK_TYPE_STATUSBAR, 
						 "RubricaStatusbar",
						 &rubrica_bar_info, 0);
    }
  
  return rubrica_bar_type;
}



static void 
rubrica_statusbar_dispose (RubricaStatusbar* self)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(self));
  
  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;  
}


static void 
rubrica_statusbar_finalize (RubricaStatusbar* self)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(self));

}


static void
rubrica_statusbar_class_init(RubricaStatusbarClass* klass)
{
  GObjectClass *object_class;
  GParamSpec* pspec;
  
  parent_class = g_type_class_peek_parent (klass);
  object_class = G_OBJECT_CLASS (klass);

  object_class->finalize = (GObjectFinalizeFunc) rubrica_statusbar_finalize;
  object_class->dispose  = (GObjectFinalizeFunc) rubrica_statusbar_dispose;

  object_class->set_property = (gpointer) rubrica_statusbar_set_property;
  object_class->get_property = (gpointer) rubrica_statusbar_get_property;
    
  g_type_class_add_private (klass, sizeof(RubricaStatusbarPrivate));


  /**
   * RubricaStatusbar:enable-extra
   *
   * 
   */
  pspec = g_param_spec_boolean("enable-extra",
			       "eneble extra",
			       "enable extra box",
			       FALSE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, STATUSBAR_EXTRA, pspec);

  /**
   * RubricaStatusbar:enable-images
   *
   * 
   */
  pspec = g_param_spec_boolean("enable-images",
			       "eneble images",
			       "enable images box",
			       FALSE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, STATUSBAR_IMAGES, pspec);

  /**
   * RubricaCardsView:msg
   *
   * the name of the addressbook displayed
   */
  pspec = g_param_spec_string("message", 
			      "message", 
			      "a message",  
			      "",
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, STATUSBAR_MESSAGE, pspec);
  

  /**
   * RubricaStatusbar:enable-rate
   *
   * 
   */
  pspec = g_param_spec_int("rate",
			   "rate pixmap",
			   "set rate pixmap",
			   PIXMAP_RATE_NONE,          // min 
			   PIXMAP_RATE_VERY_HIGHT,    // manx
			   PIXMAP_RATE_NORMAL,        // default
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, STATUSBAR_RATE, pspec);


  /**
   * RubricaStatusbar:enable-lock
   *
   * 
   */
  pspec = g_param_spec_int("lock",
			   "lock pixmap",
			   "set lock pixmap",
			   PIXMAP_UNLOCK,      // min 
			   PIXMAP_LOCK,        // max
			   PIXMAP_UNLOCK,      // default
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(object_class, STATUSBAR_LOCK, pspec);
}



static void
rubrica_statusbar_init(RubricaStatusbar* self)
{
  GtkWidget* sep;
  GtkWidget* label;
 
  g_return_if_fail(IS_RUBRICA_STATUSBAR(self));

  self->priv = RUBRICA_STATUSBAR_GET_PRIVATE(self);
 
  self->priv->extra        = gtk_hbox_new(FALSE, 0);
  self->priv->images       = gtk_hbox_new(FALSE, 0);
  self->priv->ctxid        = 0;
  self->priv->message      = NULL;
  self->priv->enable_extra = FALSE;
  self->priv->label        = gtk_label_new("");
  self->priv->rate_img     = gtk_image_new(); 
  self->priv->lock_img     = gtk_image_new();


  /* Build the statusbar
   */
  gtk_statusbar_set_has_resize_grip(GTK_STATUSBAR(self), TRUE); 
  self->priv->ctxid = gtk_statusbar_get_context_id(GTK_STATUSBAR(self), 
						   "RubricaContext"); 

  /* Build extra box
   */
  gtk_box_pack_start(GTK_BOX(self), self->priv->extra, FALSE, FALSE, 2);
  gtk_box_pack_start(GTK_BOX(self), self->priv->images, FALSE, FALSE, 2);
  gtk_widget_show(self->priv->extra);
  gtk_widget_show(self->priv->images);
  
  sep = gtk_vseparator_new();
  gtk_box_pack_start(GTK_BOX(self->priv->extra), sep, FALSE, FALSE, 2); 
  gtk_widget_show(sep);  

  gtk_box_pack_start(GTK_BOX(self->priv->extra), self->priv->label,
		     TRUE, TRUE, 2);  
  gtk_widget_show(self->priv->label);
  
  self->priv->sep = gtk_vseparator_new();
  gtk_box_pack_start(GTK_BOX(self->priv->extra), self->priv->sep, 
		     FALSE, FALSE, 2); 
  gtk_widget_hide(self->priv->sep);
  
  label = gtk_label_new(_("rate:"));
  gtk_box_pack_start(GTK_BOX(self->priv->images), label, FALSE, FALSE, 2);  
  gtk_widget_show(label);

  gtk_box_pack_start(GTK_BOX(self->priv->images), self->priv->rate_img, 
		     FALSE, FALSE, 2);    
  gtk_widget_show(self->priv->rate_img);
  
  sep = gtk_vseparator_new(); 
  gtk_box_pack_start(GTK_BOX(self->priv->images), sep, FALSE, FALSE, 2); 
  gtk_widget_show(sep); 
   
  label = gtk_label_new(_("status:"));
  gtk_box_pack_start(GTK_BOX(self->priv->images), label, FALSE, FALSE, 2);  
  gtk_widget_show(label);
  
  gtk_box_pack_start(GTK_BOX(self->priv->images), self->priv->lock_img, 
		     FALSE, FALSE, 2);  
  gtk_widget_show(self->priv->lock_img);

  gtk_widget_set_sensitive(self->priv->extra, FALSE);
  gtk_widget_set_sensitive(self->priv->images, FALSE);
  
  gtk_widget_show(GTK_WIDGET(self));

  self->priv->dispose_has_run = FALSE;
}


static void 
rubrica_statusbar_set_property (GObject* obj, guint property_id,
				GValue* value, GParamSpec* spec)
{
  RubricaStatusbar* self = (RubricaStatusbar*) obj;
  RubricaStatusbarPrivate* priv = RUBRICA_STATUSBAR_GET_PRIVATE(self);
  GdkPixbuf* pixbuf = NULL;   

  switch (property_id) 
    {
    case STATUSBAR_MESSAGE:
      if (priv->message) 
	g_free(priv->message);

      priv->message = g_value_dup_string(value);      
      
      gtk_label_set_text(GTK_LABEL(priv->label), priv->message);
      if (g_ascii_strcasecmp(priv->message, "") == 0)
	gtk_widget_hide(self->priv->sep);
      else
	gtk_widget_show(self->priv->sep);
      break;
      
    case STATUSBAR_EXTRA:
      priv->enable_extra = g_value_get_boolean(value);
      gtk_widget_set_sensitive(priv->extra, priv->enable_extra);
      break;
 
    case STATUSBAR_IMAGES:
      priv->enable_images = g_value_get_boolean(value);
      gtk_widget_set_sensitive(priv->images, priv->enable_images);
      break;

    case STATUSBAR_RATE:
      priv->rate = g_value_get_int(value);
      pixbuf = load_pixbuf(priv->rate);
      
      if (pixbuf)
	{
	  gtk_image_set_from_pixbuf (GTK_IMAGE (priv->rate_img), pixbuf);
	  gdk_pixbuf_unref (pixbuf); 
	}        
      break;
 
    case STATUSBAR_LOCK:
      priv->lock = g_value_get_int(value);
      pixbuf = load_pixbuf(priv->lock);
      
      if (pixbuf)
	{
	  gtk_image_set_from_pixbuf (GTK_IMAGE (priv->lock_img), pixbuf);
	  gdk_pixbuf_unref (pixbuf); 
	}        
      break; 

    default: 
      break; 
    } 
} 
 

static void 
rubrica_statusbar_get_property (GObject* obj, guint property_id,
				GValue* value, GParamSpec* spec)
{
  RubricaStatusbar* self = (RubricaStatusbar*) obj;
  RubricaStatusbarPrivate* priv = RUBRICA_STATUSBAR_GET_PRIVATE(self);
  
  switch (property_id) 
    {      
    case STATUSBAR_MESSAGE:
      g_value_set_string(value, priv->message);
      break;

    case STATUSBAR_EXTRA:
      g_value_set_boolean(value, priv->enable_extra);
      break;

    case STATUSBAR_IMAGES:
      g_value_set_boolean(value, priv->enable_images);
      break;

    case STATUSBAR_RATE:
      g_value_set_int(value, priv->rate);
      break;
      
    case STATUSBAR_LOCK:
      g_value_set_int(value, priv->lock);
      break;
      
    default:
      break;  
    }  
}





/*   Public
 */
/**
 * rubrica_statusbar_new
 *
 * create a #RubricaStatusbar
 *
 * returns: a new #RubricaStatusbar 
 */
GtkWidget*
rubrica_statusbar_new()
{
  GtkWidget* bar;

  bar = GTK_WIDGET(g_object_new(RUBRICA_STATUSBAR_TYPE, NULL));

  return bar;
}

void       
rubrica_statusbar_free (RubricaStatusbar* bar)
{
  g_object_unref(bar);
}



void 
rubrica_statusbar_push_message (RubricaStatusbar* bar, gchar* msg)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(bar));

  if (!msg)
    return;
  
  gtk_statusbar_push(GTK_STATUSBAR(bar), bar->priv->ctxid, msg);  
}


void 
rubrica_statusbar_clean_messages (RubricaStatusbar* bar)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(bar));

  gtk_statusbar_pop(GTK_STATUSBAR(bar), bar->priv->ctxid);
}


void 
rubrica_statusbar_enable_extra (RubricaStatusbar* bar, gboolean bool)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(bar));
  
  g_object_set(bar, "enable-extra", bool, NULL);
}

void 
rubrica_statusbar_enable_images (RubricaStatusbar* bar, gboolean bool)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(bar));
  
  g_object_set(bar, "enable-images", bool, NULL);
}


void 
rubrica_statusbar_extra_set_text (RubricaStatusbar* bar, gchar* text)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(bar));

  if(!text)
    return;
  
  g_object_set(bar, "message", text, NULL);
}


void 
rubrica_statusbar_extra_set_rate (RubricaStatusbar* bar, RubricaPixmap rate)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(bar));

  g_object_set(bar, "rate", rate, NULL);
}


void 
rubrica_statusbar_extra_set_lock (RubricaStatusbar* bar, RubricaPixmap lock)
{
  g_return_if_fail(IS_RUBRICA_STATUSBAR(bar));
 
  g_object_set(bar, "lock", lock, NULL);
}  

