/*	Data_Port

PIRL CVS ID: Data_Port.java,v 2.3 2012/04/16 06:08:56 castalia Exp

Copyright (C) 2001-2008  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import PIRL.Configuration.*;

import java.sql.Connection;
import java.util.Vector;


/** A <i>Data_Port</i> is the interface for accessing the content of a
	<i>Database</i>.
<p>
@author		Bradford Castalia, UA/PIRL
@version	2.3 
@see		Database
*/
public interface Data_Port
{
/**	Gets the Configuration of the Data_Port object.
<p>
	@return	The current Configuration of the Data_Port.
	@see	Configuration
*/
public Configuration Configuration ();

/**	Gets the required and optional parameters used by the Data_Port.
<p>
	The Configurtion that is returned will contain two groups:
	"Required" and "Optional" with lists of parameters naming those
	in each group. Each parameter may have a value indicating the
	default that will be used if the parameter is not provided in
	the Configuration used to <code>{@link #Open(Configuration)
	Open}</code> the Data_Port. A parameter without a value has no
	default; a Required parameter without a default must be in the
	provided Configuration or the <code>Open</code> will fail.
<p>
	@return	A Configuration of Required and Optional Data_Port
		parameters.
	@throws	Database_Exception	If the Data_Port could not be accessed.
	@throws	Configuration_Exception	If the database Configuration
		could not be accessed.
*/
public Configuration Parameters ()
	throws Database_Exception, Configuration_Exception;

/*==============================================================================
	Access
*/
/**	Opens the Data_Port using the parameters from the Configuration
	to access a database.
<p>
	@param	configuration	The Configuration parameters for access
		to a database through the Data_Port.
	@throws	Database_Exception	If the Data_Port could not be opened.
	@throws	Configuration_Exception	If the database Configuration
		could not be accessed.
	@see	Configuration
*/
public void Open
	(
	Configuration	configuration
	)
	throws Database_Exception, Configuration_Exception;

/**	Closes access to the database through the Data_Port.
<p>
	@throws	Database_Exception	If there was a problem releasing any of
		the resources associated with the Data_Port.
*/
public void Close ()
	throws Database_Exception;

/**	Tests if the Data_Port is currently open.
<p>
	@return	true if the Data_Port has access to a database; false
		otherwise.
*/
public boolean is_Open ();

/**	Gets the JDBC Connection object for the Data_Port, if there is one.
	There is no guarantee that any particular Data_Port will ever use a
	JDBC Connection object.
<p>
	@return	The JDBC Connection object for the Data_Port. This will be
		null if the Data_Port does not have a Connection, either
		because it does not have an active Connection or because it
		does not use a JDBC Connection .
*/
public Connection Connection ();

/**	Adds a SQL_Listener to the list of listeners to be notified of
	SQL statements before they are sent to the database server.
<p>
	@param	listener	A SQL_Listener to be added. If null, or the
		listener is already listed, nothing is done.
	@return	This Data_Port.
	@see	SQL_Listener
*/
public Data_Port Add_SQL_Listener
	(
	SQL_Listener	listener
	);

/**	Removes a SQL_Listener from the list of listeners to be notified of
	SQL statements before they are sent to the database server.
<p>
	@param	listener	A SQL_Listener to be removed.
	@return	true if the listener was removed; false if the listener was
		not listed.
	@see	SQL_Listener
*/
public boolean Remove_SQL_Listener
	(
	SQL_Listener	listener
	);

/*==============================================================================
	Description
*/
/**	Gets a one line String identifying the Data_Port.
<p>
	@return	The identifification line for the Data_Port.
*/
public String toString ();

/**	Gets a multi-line description of the Data_Port. If the Data_Port is
	currently <code>{@link #Open Open}</code>, this may include a
	description of the database connection.
<p>
	@return The multi-line description of the Data_Port and any active
		database connection.
*/
public String Description ();

/**	Gets a String describing the contents of selected catalogs
	and tables.
<p>
	For each table being described each field name and its data
	type is listed. Each Data_Port implementation may provide
	other details.
<p>
	@param	catalog	The catalog to have its contents described. If it
		is in <i>catalog</i>.<i>table</i> format, the catalog portion
		of the name will be used. If null, all catalogs on the database
		server will be described.
	@param	table	The table to have its contents described. If it is
		in <i>catalog</i>.<i>table</i> format, the table portion of the
		name will be used. If null, all tables in the catalog (or all
		catalogs) will be described.
	@return	A descriptive String.
*/
public String Contents
	(
	String	catalog,
	String	table
	);

/**	Gets the list of catalogs on the database server.
<p>
	@return	A Vector of catalog name Strings.
	@throws	Database_Exception	If the Data_Port is not open, or
		the operation on the database server failed.
*/
public Vector<String> Catalogs ()
	throws Database_Exception;

/**	Gets the list of tables in a catalog on the database server.
<p>
	@param	catalog	The name of the catalog to be examined. If null,
		the <code>{@link Database#CATALOG CATALOG}</code> from the
		Data_Port <code>Configuration</code> will be used.
	@return	A Vector of table name Strings. If the catalog does not
		exist in the database, null will be returned. An empty Vector
		will be returned for a catalog that does not contain any
		tables.
	@throws	Database_Exception	If the Data_Port is not open, or
		the operation on the database server failed.
*/
public Vector<String> Tables
	(
	String	catalog
	)
	throws Database_Exception;

/**	Gets the list of field names in a table on the database server.
<p>
	@param	table	The name of the table to be examined. If null, then
		the <code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in
		<i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@return	A Vector of field name Strings. If the table does not exist
		in the Database, null will be returned.
	@throws	Database_Exception	If the Data_Port is not open, no
		catalog name is available, or the operation on the database
		server failed.
*/
public Vector<String> Field_Names
	(
	String	table
	)
	throws Database_Exception;

/**	Gets the list of field data types in a table on the database
	server.
<p>
	@param	table	The name of the table to be examined. If null, then
		the <code>{@link Database#TABLE TABLE}</code> from the
		<code>Configuration</code> will be used. If it is not in
		<i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@return	A Vector of field type name Strings. If the table does not exist
		in the Database, null will be returned.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public Vector<String> Field_Types
	(
	String	table
	)
	throws Database_Exception;

/**	Obtains a list of field information for the table of a catalog.
<p>
	The field_info tag corresponding to one of the field information
	names known to the database server. The possible tag lists should
	be available from the description of the particular Data_Port
	implementation being used, or in the database server documentation.
<p>
	@param	table	The name of the table to be examined. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@param	field_info	The tag String for selecting field information.
	@return	A Vector of field information Strings, one per field. For
		info that is originally numeric (as indicated in the table) a
		String representation is returned. If the table does not exist
		in the Database, a null will be returned.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public Vector<String> Fields
	(
	String	table,
	String	field_info
	)
	throws Database_Exception;

/**	Gets a list of primary keys for a table.
<p>
	@param	table	The name of the table to be examined. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@return	A Vector of field name Strings.
	@throws	Database_Exception	If no catalog or table name is
		available, or the database server rejected the operation.
*/
public Vector<String> Keys
	(
	String	table
	)
	throws Database_Exception;

/*==============================================================================
	Query
*/
/**	Submit an SQL query.
<p>
	The SQL_query String is an SQL statment that requests data to be
	extracted (not removed) from the database. The syntax, though
	"standardized", is likely to vary in its particulars dependent
	on the database server implementation.
<p>
	A query always returns a table. The table is in the form of a
	Vector of data record Vectors. The first record contains the names
	(String) of each field for the data in corresponding locations of
	all the field value records that follow; i.e. these are the table
	column names. The field values in each data record are always
	provided in String representation regardless of the data type
	stored in the database. However, NULL field values remain unchanged
	as they are distinct from an otherwise valid field value.
<p>
	<b>Note</b>: A field value will either be a String or null.
<p>
	@param	SQL_query	The syntax of the query string is database dependent.
		A typical example is an SQL "SELECT" statement.
	@param	limit	The maximum number of records to return. If
		negative, there will be no limit to the number of records
		returned. If zero, no records will be returned.
	@return	A data table in form of a Vector of String Vectors (records).
		The first record is the field names for the field values in all
		subsequent records.
	@throws	Database_Exception	If the Data_Port is not open or the
		operation on the database server failed.
*/
public Vector<Vector<String>> Query
	(
	String	SQL_query,
	int		limit
	)
	throws Database_Exception;

/**	Selects from one or more tables the data values from one or more
	named fields from the data records that satisfy the selection
	conditions.
<p>
	<b>Note</b>: Each table name that is not in the format
	<i>catalog</i>.<i>table</i> will have the <code>{@link
	Database#CATALOG CATALOG}</code> from the
	<code>Configuration</code> prepended.
<p>
	@param	tables	The Vector of database tables from which to select
		records. If this is null, the table specified in the Configuration
		will be used.
	@param	fields	A Vector of field name Strings specifying the
		data fields (columns) to be extracted. If this is null, all
		data fields will be extracted.
	@param	conditions	A String in SQL WHERE clause syntax
		(without the keyword "WHERE") specifying the conditions for
		selection of a data record from the database. The specific
		syntax of the conditions string is database dependent. If
		this is null, no conditions will be applied; all data records
		will be used.
	@param	limit	The maximum number of records to return. If
		negative, there will be no limit to the number of records
		returned. If zero, the returned vector will contain a single
		string that is the SQL statement that would have been submitted
		to the Query method.
	@return	A Vector of String Vectors data table,as provided by the
		<code>{@link #Query(String, int) Query}</code> method.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public Vector<Vector<String>> Select
	(
	Vector<String>	tables,
	Vector<String>	fields,
	String			conditions,
	int				limit
	)
	throws Database_Exception;

/*==============================================================================
	Update
*/
/**	Submit an SQL update.
<p>
	The syntax of the SQL_update String is database dependent. These
	operations modify the database and return a count of the number of
	modifications, which may be 0 in some cases (e.g. when a catalog or
	table is created). Typical examples are SQL "UPDATE", "INSERT", and
	"ALTER" statements.
<p>
	@param	SQL_update	The SQL_update statement.
	@return	A count of the number of modifications.
	@throws	Database_Exception	If the Data_Port is not open or the
		operation on the database server failed.
*/
public int Update
	(
	String	SQL_update
	)
	throws Database_Exception;

/*------------------------------------------------------------------------------
	Catalogs:
*/
/**	Creates a catalog in the database.
<p>
	@param	catalog	The name of the catalog to create. A null catalog
		name, or a name of a catalog already in the database, is ignored.
	@throws	Database_Exception	If the Data_Port is not open or the
		operation on the database server failed.
*/
public void Create
	(
	String	catalog
	)
	throws Database_Exception;

/**	Deletes a catalog from the database.
<p>
	@param	catalog	The name of the catalog to delete. A null catalog
		name, or a name that is not a catalog in the database, is ignored.
	@throws	Database_Exception	If the Data_Port is not open or the
		operation on the database server failed.
*/
public void Delete
	(
	String	catalog
	)
	throws Database_Exception;

/*------------------------------------------------------------------------------
	Tables:
*/
/**	Creates data tables with their fields and the data types of the fields.
<p>
	If the table does not exist, a new one is created. For each name
	String in the fields Vector, if the field does not exist, it is
	created with the data type from the corresponding entry in the
	types Vector. If the field is already present in the table, its
	data type is changed if needed.
<p>
	A table name that is not in the format <i>catalog</i>.<i>table</i>
	will have the <code>{@link Database#CATALOG CATALOG}</code>
	from the <code>Configuration</code> prepended.
<p>
	@param	table	The name of the table to be created. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names to be used.
	@param	types	The Vector of data type names to be applied to
		the corresponding fields.
	@throws	Database_Exception	If the Data_Port is not open, the
		number of fields and types are not the same, no table name is
		available, or the operation on the database server failed.
	@see	#Configuration()
*/
public void Create
	(
	String			table,
	Vector<String>	fields,
	Vector<String>	types
	)
	throws Database_Exception;

/**	Deletes fields from a data table, or the entire table.
<p>
	Each field in the fields list is deleted from the table. If
	the field is not present in the table, it is ignored. If the
	fields list is null, the entire table is deleted from the
	catalog. <b>N.B.</b>: A table is not deleted even if all of
	its fields are deleted.
<p>
	@param	table	The name of the table to be deleted. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names to be used.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
	@see	#Configuration()
*/
public void Delete
	(
	String			table,
	Vector<String>	fields
	)
	throws Database_Exception;

/**	Renames a table.
<p>
	If the table does not exist, nothing is done.
<p>
	@param	table	The name of the table to be renamed. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@param	name	The new name for the table. If this is null,
		nothing is done.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public void	Rename
	(
	String	table,
	String	name
	)
	throws Database_Exception;

/**	Renames fields in a table.
<p>
	Each field name in the fields Vector that exists in the table is
	renamed to the corresponding name in the names Vector. If the
	table does not exist, nothing is done.
<p>
	<b>Note</b>: The existing type for the field, obtained from the
	<code>{@link #Field_Types(String) Field_Types}</code> method, is
	re-applied to the field (due to requirments of the SQL syntax).
	This could alter other characteristics of the field.
<p>
	@param	table	The name of the table to be affected. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names to be renamed.
	@param	names	The Vector of new field names.
	@throws	Database_Exception	If the Data_Port is not open, the
		number of fields and names are not the same, no table name is
		available, or the operation on the database server failed.
*/
public void Rename
	(
	String			table,
	Vector<String>	fields,
	Vector<String>	names
	)
	throws Database_Exception;

/*------------------------------------------------------------------------------
	Fields:
*/
/**	Inserts values for selected fields into a table.
<p>
	A new data record is created in the table. The fields Vector lists
	the names of the fields to be assigned a data value from the
	corresponding entry in the values Vector. The Vector of data values
	does not necessarily have to contain Strings, as long as each
	object's toString method produces a valid representation of the
	field value. Though the fields and values Vectors must be the same
	size, not all fields in the table need to be included; the database
	is expected to provide the default for missing fields.
<p>
	@param	table	The name of the table to be affected. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names for the data values.
	@param	values	The Vector of data values corresponding the Vector
		of field names.
	@return	The number of records inserted; 1 if successful; 0 otherwise.
	@throws	Database_Exception	If the Data_Port is not open, the
		number of fields and values are not the same, no table name is
		available, or the operation on the database server failed.
*/
public int Insert
	(
	String			table,
	Vector<String>	fields,
	Vector<Object>	values
	)
	throws Database_Exception;

/**	Updates values for selected fields into a table.
<p>
	The fields Vector lists the names of the fields to be assigned a
	new data value from the corresponding entry in the values Vector.
	The Vector of data values does not necessarily have to contain
	Strings, as long as each object's toString method produces a valid
	representation of the field value. The records to have their field
	values updated will be selected by the conditions expression. This
	is an SQL WHERE expression (without the "WHERE" keyword). If no
	conditions are provided, then all records will be updated.
<p>
	@param	table	The name of the table to be affected. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@param	fields	The Vector of field names for the data values.
	@param	values	The Vector of data values corresponding the Vector
		of field names.
	@param	conditions	A String in SQL WHERE clause syntax
		(without the keyword "WHERE") specifying the conditions for
		selection of a data record from the database. The specific
		syntax of the conditions string is database dependent. If
		this is null, no conditions will be applied; all data records
		will be used.
	@return	The number of records that were modified.
	@throws	Database_Exception	If the Data_Port is not open, the
		number of fields and values are not the same, no table name is
		available, or the operation on the database server failed.
*/
public int Update
	(
	String			table,
	Vector<String>	fields,
	Vector<Object>	values,
	String			conditions
	)
	throws Database_Exception;

/**	Deletes selected records from a table.
<p>
	The records to be deleted will be selected by the conditions
	expression. This is an SQL WHERE expression (without the "WHERE"
	keyword).
<p>
	<b>Warning</b>: If no conditions are provided, then all records will
	be deleted leaving an empty table.
<p>
	@param	table	The name of the table to be affected. If this is
		null then the <code>{@link Database#TABLE TABLE}</code>
		from the <code>Configuration</code> will be used. If it is not
		in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> will be used.
	@param	conditions	A String in SQL WHERE clause syntax
		(without the keyword "WHERE") specifying the conditions for
		selection of a data record from the database. The specific
		syntax of the conditions string is database dependent. If
		this is null, no conditions will be applied and all data records
		will be deleted.
	@return	The number of records that were deleted.
	@throws	Database_Exception	If the Data_Port is not open, no table
		name is available, or the operation on the database server
		failed.
*/
public int Delete
	(
	String	table,
	String	conditions
	)
	throws Database_Exception;

/*==============================================================================
	Utility
*/
/**	Generates a table reference from a catalog and table name.
<p>
	A table reference in a Data_Port method may be a composite name
	combining a catalog name and a table name. The specific format for
	this combination is dependent on the syntax requirements of the
	database being used.
<p>
	If a non-null, non-empty catalog name is specified it will override
	any catalog name portion that might be in a table name that is
	already in table reference format. The Data_Port may use
	appropriate defaults for null or empty table or catalog names; e.g.
	the value of <code>{@link Database#CATALOG CATALOG}</code> and
	<code>{@link Database#TABLE TABLE}</code> Configuration
	parameters. Though how defaults, if any, are provided is at the
	discretion of the Data_Port.
<p>
	@param	catalog	The catalog name portion for the table reference.
	@param	table	The table name portion for the table reference.
	@return	A table reference suitable for use as a <code>table</code>
		argument of a Data_Port method.
	@throws	Database_Exception	If a table reference can not be formed.
*/
public String Table_Reference
	(
	String	catalog,
	String	table
	)
	throws Database_Exception;

/**	Gets the catalog name portion of a table reference.
<p>
	@param	table_reference	A String that may be a composite name
		combining a catalog name and a table name in a database
		specific format.
	@return	The catalog name portion of the table reference. If the
		table reference does not have a catalog name portion the empty
		String is returned.
	@throws	Database_Exception	If the Data_Port is not open.
	@see #Table_Reference(String, String)
*/
public String Catalog_Name
	(
	String	table_reference
	)
	throws Database_Exception;

/**	Gets the name known to the database server for a catalog name.
<p>
	Only accessible catalogs can be identified. If the database server
	determines that the connection does not have permission to access
	the specified catalog then it can not be identified.
<p>
	<b>N.B.</b>: Identifying the catalog from the list of accessible
	catalogs is done using the {@link #Case_Sensitive_Identifiers() case
	sensitivity} of the Data_Port implementation. Therefore, if case
	sensitive matching is used this method can only return the specified
	catalog name or null; in this case this method is only useful for
	determining if a catalog is accessible or not.
<p>
	@param	catalog	The name of the database calolog to examine. If this
		is a {@link #Table_Reference(String, String) table reference}
		only the catalog part will be used. If null or empty the
		<code>{@link Database#CATALOG CATALOG}</code> value from the
		{@link #Configuration() configuration}, if available, will be
		used.
	@return	The name of the catalog as it is known in the database. This
		will be null if a non-null, non-empty catalog name could not be
		identified in the database.
	@throws	Database_Exception	If the Data_Port is not open, a catalog
		name was not provided, or the database rejects the operation.
*/
public String Database_Catalog_Name
	(
	String	catalog
	)
	throws Database_Exception;

/**	Gets the table name portion of a table reference.
<p>
	@param	table_reference	A String that may be a composite name
		combining a catalog name and a table name in a database
		specific format.
	@return	The table name portion of the table reference. If the table
		reference is null the empty String will be returned.
	@throws	Database_Exception	If the Data_Port is not open.
	@see #Table_Reference(String, String)
*/
public String Table_Name
	(
	String	table_reference
	)
	throws Database_Exception;

/**	Gets the name known to the database server for a table name.
<p>
	Only accessible catalogs can be identified. If the database server
	determines that the connection does not have permission to access
	the specified catalog then it can not be identified.
<p>
	<b>N.B.</b>: Identifying the catalog from the list of accessible
	catalogs is done using the {@link #Case_Sensitive_Identifiers() case
	sensitivity} of the Data_Port implementation. Therefore, if case
	sensitive matching is used this method can only return the specified
	catalog name or null; in this case this method is only useful for
	determining if a catalog is accessible or not.
<p>
	@param	table	The name of the table to identify in the database. If
		null, then the <code>{@link Database#TABLE TABLE}</code> value
		from the <code>Configuration</code>, if present, will be used. If
		it is not in <i>catalog</i>.<i>table</i> format, the <code>{@link
		Database#CATALOG CATALOG}</code> value, if present, will be used.
	@return	The name of the catalog as it is known in the database. This
		will be null if a catalog name could not be identified in the
		database.
	@throws	Database_Exception	If the Data_Port is not open, catalog and
		table names were not provided, or the database rejects the
		operation.
*/
public String Database_Table_Name
	(
	String	table
	)
	throws Database_Exception;

/**	Gets the table reference component delimiter.
<p>
	@return	The portion of a table reference that delimits the
		catalog component from the table component.
	@throws	Database_Exception	If the Data_Port is not open.
	@see #Table_Reference(String, String)
*/
public String Table_Reference_Component_Delimiter ()
	throws Database_Exception;

/**	Tests if identifier names are treated as case sensitive by the
	Data_Port.
<p>
	Because some database servers on some operating systems are not case
	sensitive in handling identifiers (the names of catalogs, tables, and
	field names) it is necessary to enforce case insensitivity when
	matching user specified names against identifiers returned from the
	database server. This method indicates how the Data_Port expects
	identifier names to be treated.
<p>
	<b>N.B.</b>: The identifier case sensitivity that is reported by the
	Data_Port does not necessarily reflect the identifier case
	sensitivity of the database server under all circumstances. Some
	identifiers may be case sensitive and others not, a given identifier
	may be case sensitive on some systems but not on others, an
	identifier may be case sensitive if double quoted but otherwise not
	(PostgreSQL); the rules are database implementation specific.
<p>
	What the Case_Sensitive_Identifiers flag indicates is how indentifiers
	provided the by the user will be matched against identifiers provided
	by the database server.
<p>
	@return	true if identifiers will be matched with case sensitive
		comparisons; false if matching will ignore case.
	@see	Database#Matches(String, String)
*/
public boolean Case_Sensitive_Identifiers ()
	throws Database_Exception;

/**	Attempts to set whether identifier names are treated as case
	sensitive by the Data_Port.
<p>
	<b>N.B.</b>: This setting does not itself determine the case
	sensitivity of the identifiers by the database server. The value is
	used to inform the Database software if case sensitive matches should
	be done. Setting the value to true when the database server uses case
	insensitive identifiers (mixed case identifiers provided by the user
	to the database server being always read back as all lowercase) will
	ultimately result in unexpected mismatches. Only set the value to
	true when it is known that the database server accessed by the
	Data_Port is capable of using case sensitive identifiers.
<p>
	<b>N.B.</b>: A Data_Port implementation is free to ignore this
	setting, and may even through an exception if it thinks that the
	value would cause problems. In general, it is best to leave the case
	sensitive identifiers setting up to the Data_Port. Only when the
	application is aware of particular circumstances when using a
	particular database server that would call for a change of the
	Data_Port setting should the value be changed.
<p>
	@param	case_sensitive	true if identifiers are to be matched with
		case sensitive comparisons; false if matching is to ignore case.
	@throws	Database_Exception	If the Data_Port can not be accessed,
		or the Data_Port rejects the setting.
	@see	#Case_Sensitive_Identifiers()
*/
public void Case_Sensitive_Identifiers
	(
	boolean		case_sensitive
	)
	throws Database_Exception;

}


