/*	Configuration

PIRL CVS ID: Configuration.java,v 1.55 2012/04/16 06:05:20 castalia Exp

Copyright (C) 2003-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Configuration;

import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.PVL.Parser;
import	PIRL.PVL.Lister;
import	PIRL.PVL.Selector;
import	PIRL.PVL.Selection;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Conductor.Reference_Resolver;
import	PIRL.Utilities.Host;
import	PIRL.Strings.String_Buffer;

import	java.lang.StringBuffer;
import	java.util.Vector;
import	java.util.Stack;
import	java.util.Hashtable;
import	java.util.Properties;
import	java.util.Enumeration;
import	java.util.ListIterator;
import	java.util.Iterator;
import	java.io.File;
import	java.io.FileInputStream;
import	java.io.StringWriter;
import	java.io.InputStream;
import	java.io.IOException;
import	java.io.FileNotFoundException;
import	java.net.URL;
import	java.net.MalformedURLException;
import	java.text.ParseException;
import	java.util.regex.Pattern;


/**	A <I>Configuration</I> maintains a <I>Parameter</I> list used for
	configuration management.
<P>
	A Configuration is constructed by parsing source text in Parameter
	Value Language (PVL) format (as read by a Parameter {@link
	PIRL.PVL.Parser Parser}), or from another Configuration or PVL
	Parameter. The {@link #Default_Source() Default_Source} will be
	used if no source is specified. Except when a Configuration is
	constructed as a copy of another Configuration (or Parameter
	Aggregate), the {@link #Defaults Defaults} Parameters are always
	included; if the special name {@link #DEFAULTS DEFAULTS} is
	specified, only the Defaults Parameters will be in the
	Configuration. In all cases, the Parameters in the Configuration
	will be {@link #Coalesce() Coalesce}d together.
<P>
	The parameters of a Configuration are refered to by their
	<I>pathname</I>. The pathname may include a {@link
	Parameter#Path_Delimiter() delimiter} (usually the '/' character)
	separated sequence of names where each segment names a parameter
	group within which subsequent named parameters are nested. A
	pathname that begins with a delimiter is rooted at the top of the
	Configuration parameter groupings (i.e. these pathnames have an
	absolute location in the Configuration). Otherwise the pathname is
	rooted at the first occurance of a parameter having the first
	segment name (i.e. these pathnames have a relative location in the
	Configuration).
<P>
	There are a set of parameter names that are known to the
	Configuration package:
<P>
<DL>
<DT><B>{@link #CLASSPATH Classpath}</B>
	<DD>The Java CLASSPATH. The default is obtained from the
		"java.class.path" System Property.
<DT><B>{@link #HOST Host}</B>
	<DD>An application-specific hostname. The default is "localhost."
<DT><B>{@link #USER User}</B>
	<DD>A username to be supplied to the application. The default is
		obtained from the "user.name" System Property.
<DT><B>{@link #ALIAS Alias}</B>
	<DD>The name of the group from which a new Configuration was
		created when an alias (a linked parameter) was used rather
		than a group name.
</DL>
<P>
	In addition, the {@link #INCLUDE} parameter is recognized when
	{@link #Include(boolean) Include} mode is enabled as a parameter
	with a value that references another source of Parameters that is to
	be included in the Configuration replacing the Include parameter.
	Include files may be nested. Cyclical Include file (but not URL)
	chains are caught.
<P>
@see		PIRL.PVL.Parameter

@author		Bradford Castalia - UA/PIRL
@version	1.55 
*/
public class Configuration
	extends Parameter
	implements Cloneable
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Configuration.Configuration (1.55 2012/04/16 06:05:20)";

/**	The name of the configuration source from which the parameters have
	been set.
<P>
	This will be {@link #DEFAULTS DEFAULTS} if the Configuration was
	constructed soley from the {@link #Defaults Defaults} Parameters. It
	will be null if it was not constructed from the contents of a file
	(i.e. it was constructed from a Parameter).
*/
private String
	Config_Source		= null;

/**	The parameter name for the Java CLASSPATH: "Classpath".
*/
public static final String
	CLASSPATH			= "Classpath";

/**	The parameter name for the application-specific hostname: "Host".
*/
public static final String
	HOST				= "Host";

/**	The parameter name for the application-specific username: "User".
*/
public static final String
	USER				= "User";

/**	The parameter name for a linked parameter used to create a new
	Configuration: "Alias".
<P>
	@see		#Group(String)
*/
public static final String
	ALIAS				= "Alias";

/**	The name of the Group containing the environment variable
	parameters: "/Environment".
<P>
	@see	#Add_Environment()
*/
public static final String
	ENVIRONMENT			= Path_Delimiter () + "Environment";

/**	The name of a parameter that refers to a file of parameters to be
	included at that point in the Configuration: "@Include".
*/
public static final String
	INCLUDE				= "@Include";

/**	The default automatic {@link #Include(boolean) Include} state.
*/
public static boolean
	Include_Default		= true;
private boolean
	Include_Includes	= Include_Default;

/**	Parameter name prefix for the start of an {@link #Include(boolean)
	Include} file when tracing is enabled.
*/
public static final String
	INCLUDE_START		= "INCLUDE_START";

/**	Parameter name prefix for an {@link #CONTINUE_INCLUDE_ON_ERROR_FLAG
	ignored error} report from an {@link #Include(boolean) Include} file
	when tracing is enabled.
*/
public static final String
	INCLUDE_ERROR		= "INCLUDE_ERROR";

/**	Parameter name prefix for the end of an {@link #Include(boolean)
	Include} file when tracing is enabled.
*/
public static final String
	INCLUDE_END			= "INCLUDE_END";

/**	The default automatic {@link #Include(boolean) Include} tracing flag.
*/
public static boolean
	Include_Tracing_Default	= false;
private boolean
	Include_Tracing		= Include_Tracing_Default;


/**	{@link #INCLUDE} parameter value flag that, when
	found at the beginning of the value, causes an error on the
	include - the source does not exists - to be ignored and
	the include skipped.
*/
public static final String
	CONTINUE_INCLUDE_ON_ERROR_FLAG = "-";

/**	The name of the internal default parameters: "DEFAULTS".
*/
public static final String
	DEFAULTS			= "DEFAULTS";

/**	The name of the default configuration source: initially "DEFAULTS".
*/
public static String
	Default_Source		= DEFAULTS;

/**	The default parameters conditionally set during {@link
	#Configure(InputStream) configuration}.
<P>
	The default parameters are:
	<DL>
	<DT>{@link #CLASSPATH CLASSPATH}
		<DD>The "java.class.path" System Property.
	<DT>{@link #USER USER}
		<DD>The "user.name" System Property.
	<DT>{@link #HOST HOST}
		<DD>The "localhost" String.
	</DL>
*/
public static final Parameter
	Defaults			= new Parameter (DEFAULTS);
static
	{
	try
		{
		Defaults.Add (new Parameter (CLASSPATH)
			.Value (System.getProperty ("java.class.path")));
		Defaults.Add (new Parameter (USER)
			.Value (System.getProperty ("user.name")));
		Defaults.Add (new Parameter (HOST)
			.Value ("localhost"));
		}
	catch (PVL_Exception exception) {/*	This shouldn't happen. */}
	}

/**	The default automatic {@link #Defaults(boolean) Defaults} state.
*/
public static boolean
	Defaults_Default	= true;

private boolean
	Include_Defaults	= Defaults_Default;


private static Class
	Relative_to_Class	= null;


/**	The first duplicate parameter in the list gets preference when
	coalescing.
<P>
	When a duplicate Assignment parameter pathname is found while
	{@link #Coalesce() coalescing} the Configuration, give preference
	to the first parameter; i.e. remove all but the first of any
	duplicate pathname parameters.
*/
public static final int
	PREFER_FIRST_PARAMETER			= 1;

/**	The last duplicate parameter in the list gets preference when
	coalescing.
<P>
	When a duplicate Assignment Parameter pathname is found while
	{@link #Coalesce() coalescing} the Configuration, give preference
	to the last parameter; i.e. remove all but the last of any
	duplicate pathname parameters.
*/
public static final int
	PREFER_LAST_PARAMETER			= -1;

/**	Duplicate parameters in the list when coalescing causes an
	exception.
<P>
	When a duplicate Assignment Parameter pathname is found while
	{@link #Coalesce() coalescing} the Configuration, a
	Configuration_Exception will be thrown.
*/
public static final int
	THROW_ON_DUPLICATE_PARAMETERS	= 0;

/**	The default {@link #Duplicate_Parameter_Action()}.
*/
public static int
	Duplicate_Parameter_Action_Default
		= THROW_ON_DUPLICATE_PARAMETERS;

private int
	Duplicate_Parameter_Action
		= Duplicate_Parameter_Action_Default;

/**	The default {@link #Case_Sensitive()} state.
*/
public static boolean
	Case_Sensitive_Default			= false;

/**	Is parameter name matching case sensitive?
*/
private boolean
	Case_Sensitive					= Case_Sensitive_Default;


public static final String
	NL								= System.getProperty ("line.separator");


//	DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_GET		= 1 << 0,
	DEBUG_SET		= 1 << 1,
	DEBUG_CONFIGURE	= 1 << 2,
	DEBUG_INCLUDE	= 1 << 3,
	DEBUG_UTILITY	= 1 << 4,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a default Configuration.
<P>
	The {@link #Defaults} are used to set the initial parameters. {@link
	#DEFAULTS} is the name of this Configuration.
*/
public Configuration ()
{
try {Configure (DEFAULTS);}
catch (Configuration_Exception exception)
	{/*	This shouldn't happen if the Defaults are OK. */}
}

/**	Constructs a Configuration using the specified source.
<P>
	@param	source	The source String reference for the Configuration
		PVL. This may be in the form of a URL representation. If null,
		the {@link #Default_Source() Default_Source} is used.
	@throws	IllegalArgumentException	If the source is not a URL and
		no file or system resource can be found with the source name.
	@throws	Configuration_Exception	If the source can not be read or
		successfully parsed for parameters, {@link #INCLUDE} files could
		not be included, or there was a problem coalescing the
		parameters (probably due to a duplicate parameter name).
	@see	#Configure(String)
	@see	#Include(boolean)
*/
public Configuration
	(
	String	source
	)
	throws Configuration_Exception
{Configure (source);}

/**	Constructs a Configuration using the specified URL.
<P>
	@param	url		A URL object to use as the source of Configuration
		PVL.
	@throws	Configuration_Exception	If a stream can not be opened from
		the URL, there was a problem while parsing the source for
		parameters, {@link #INCLUDE} files could not be included, or
		there was a problem coalescing the parameters (probably due to a
		duplicate parameter name).
	@see	#Configure(URL)
	@see	#Include(boolean)
*/
public Configuration
	(
	URL		url
	)
	throws Configuration_Exception
{Configure (url);}

/**	Constructs a Configuration using the specified InputStream.
<P>
	@param	stream	An InputStream object to use as the source of
		Configuration PVL.
	@throws	Configuration_Exception	If there was a problem while parsing
		the stream for parameters, {@link #INCLUDE} files could not be
		included, or there was a problem coalescing the parameters
		(probably due to a duplicate parameter name).
	@see	#Configure(InputStream)
	@see	#Include(boolean)
*/
public Configuration
	(
	InputStream	stream
	)
	throws Configuration_Exception
{Configure (stream);}

/**	Constructs a Configuration from a Parameter.
<P>
	If the Parameter is null or empty (has the {@link Parameter#UNKNOWN}
	classification) an empty Configuration with the name {@link
	Parser#CONTAINER_NAME} is constructed. If the Parameter is an
	Assignment it becomes the only parameter in a Configuration with the
	name {@link Parser#CONTAINER_NAME}. If the Parameter is an
	Aggregate, its Parameter list is copied into the new Configuration
	which is given the name of the Aggregate (the Configuration is a
	copy of the Parameter).
<P>
	If automatic include is enabled any {@link #INCLUDE} parameters are
	processed. Finally, the results are {@link #Coalesce() Coalesce}d.
<P>
	<B>N.B.</B>: Only the provided Parameter is included in the new
	Configuration; Defaults are not included.
<P>
	@param	parameter	The Parameter which will be copied into this
		Configuration.
	@throws	Configuration_Exception	If the Parameter contents could not
		be added to the new Configuration, {@link #INCLUDE} files could
		not be included, or there was a problem coalescing the results
		(probably due to duplicate parameter names).
	@see	#Coalesce()
	@see	#Include(boolean)
*/
public Configuration
	(
	Parameter	parameter
	)
	throws Configuration_Exception
{
if (parameter != null)
	{
	try
		{
		if (parameter.Is_Aggregate ())
			{
			Data (parameter);
			Name (parameter.Name ());
			}
		else
			{
			Add (new Parameter (parameter));
			Name (Parser.CONTAINER_NAME);
			}
		if (Include_Includes)
			Include ();
		Coalesce ();
		}
	catch (PVL_Exception exception)
		{
		throw new Configuration_Exception
			(
			ID + NL
			+ "Failed to construct a new Configuration from the \""
				+ parameter.Name () + "\" Parameter."
			+ exception_message (exception),
			exception
			);
		}
	}
else
	Ensure_Aggregate_Configuration ();
}

/**	Copies a Configuration.
<P>
	This is the same as constructing a Configuration from a Parameter,
	followed by copying over the Configuration characteristics {@link
	#Case_Sensitive(boolean) case sensitivity}, {@link
	#Duplicate_Parameter_Action(int) duplicate parameter handling},
	{@link #Include(boolean) Include processing}, {@link
	#Include_Tracing(boolean) Include tracing}, {@link #Defaults(boolean)
	Defaults parameters inclusion} and the {@link #Source() filename} if
	non-null;
<P>
	<b>N.B.</B>: The state of this Configuration is used during the
	copy, not the state of the copied Configuration. For example, copied
	{@link #INCLUDE} parameters are processed if automatic {@link
	#Include(boolean) Include} is enabled for this Configuration; the
	automatic Include state of the Configuration being copied is not
	used.
<P>
	@param	configuration	The Configuration to be copied.
	@throws	Configuration_Exception	If the Parameter contents could not
		be added to the new Configuration, or there was a problem
		coalescing the results (probably due to duplicate parameter
		names).
	@see	#Configuration(Parameter)
*/
public Configuration
	(
	Configuration configuration
	)
	throws Configuration_Exception
{
this ((Parameter)configuration);
Config_Source				= configuration.Config_Source;
Include_Includes			= configuration.Include_Includes;
Include_Tracing				= configuration.Include_Tracing;
Include_Defaults			= configuration.Include_Defaults;
Duplicate_Parameter_Action	= configuration.Duplicate_Parameter_Action;
Case_Sensitive				= configuration.Case_Sensitive;
}

/**	Clones a Configurtion.
<P>
	@return	An Object of class Configuration that is a copy of this
		Configuration. This will be null if there was a problem making
		the copy.
	@see	#Configuration(Configuration)
	@see	Object#clone()
	@see	Cloneable
*/
public Object clone ()
{
try {return new Configuration (this);}
catch (Configuration_Exception exception) {return null;}
}

/*==============================================================================
	Accessors
*/
/**	Gets the value(s) for a parameter at a pathname.
<P>
	If specific is true then the values will be obtained from the {@link
	#Specific_Parameter(String) Specific_Parameter}, otherwise the
	{@link #Effective_Parameter(String) Effective_Parameter} will be
	used.
<P>
	@param	pathname	A parameter pathname.
	@param	specific	true if only a parameter at the specific
		pathname is acceptable, otherwise default values from an
		effective parameter will do. 
	@return	A Vector of zero or more values. Each element of the
		Vector is either the String representation of a single
		value, or is itself a Vector of the same kind.
*/
public Vector Get
	(
	String	pathname,
	boolean	specific
	)
{return Get (Parameter (pathname, specific));}

/**	Gets the value(s) for a parameter at the Nth pathname.
<P>
	@param	pathname	A parameter pathname.
	@param	specific	true if only a parameter at the specific
		pathname is acceptable, otherwise default values from an
		effective parameter will do.
	@param	skip		The number of matching parameters to skip
		before selecting a parameter.
	@return	A Vector of zero or more values. Each element of the
		Vector is either the String representation of a single
		value, or is itself a Vector of the same kind.
	@see	#Get(String, boolean)
	@see	#Parameter(String, boolean, int)
*/
public Vector Get
	(
	String	pathname,
	boolean	specific,
	int		skip
	)
{return Get (Parameter (pathname, specific, skip));}

/**	Gets the value(s) for a parameter at a specific pathname.
<P>
	This is the same as using the {@link #Get(String, boolean) Get}
	method with the specific argument being true.
<P>
	@param	pathname	A parameter pathname.
	@return	A Vector of zero or more values. Each element of the
		Vector is either the String representation of a single
		value, or is itself a Vector of the same kind.
*/
public Vector Get_Specific
	(
	String	pathname
	)
{return Get (pathname, true);}

/**	Gets one String value from the specific pathname.
<P>
	@param	pathname	A parameter pathname.
	@return	The first String value from the specific parameter, or null
		if there is no parameter value.
	@see	#Get_Specific(String)
	@see	#Get_One(Vector)
*/
public String Get_Specific_One (String pathname)
{return Get_One (Get (pathname, true));}

/**	Gets the value(s) for the effective parameter at a pathname.
<P>
	This is the same as using the {@link #Get(String, boolean) Get}
	method with the specific argument being false.
<P>
	@param	pathname	A parameter pathname.
	@return	A Vector of zero or more values. Each element of the
		Vector is either the String representation of a single
		value, or is itself a Vector of the same kind.
*/
public Vector Get
	(
	String	pathname
	)
{return Get (pathname, false);}

/**	Gets one String value for the effective parameter at a pathname.
<P>
	@param	pathname	A parameter pathname.
	@return	The first String value from the effective parameter, or
		null if there is no value.
	@see	#Get(String)
	@see	#Get_One(Vector)
*/
public String Get_One (String pathname)
{return Get_One (Get (pathname, false));}

/**	Gets the value(s) for the effective parameter at a pathname
	where the parameter may be linked to another parameter through
	its value.
<P>
	A linked parameter is an assignment of a single String value in
	which the value is the pathname of some other effective parameter.
	If the parameter has more than one value it is not linked. The
	returned Vector contains the value(s) of the first non-linked
	parameter found. An empty Vector will be returned if no parameter is
	found.
<P>
	@param	pathname	A parameter pathname.
	@return	A Vector of zero or more values. Each element of the
		Vector is either the String representation of a single
		value, or is itself a Vector of the same kind.
*/
public Vector Get_Linked
	(
	String	pathname
	)
{
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println (">>> Configuration.Get_Linked: \"" + pathname + "\"");
Vector
	value = Get (pathname),
	link;
while (value.size () == 1 &&
		(link = Get (Get_One (value))).size () != 0)
	{
	if ((DEBUG & DEBUG_GET) != 0)
		System.out.println ("    Linked to: " + link);
	value = link;
	}
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println ("<<< Configuration.Get_Linked: " + value);
return value;
}

/**	Gets one String value for the effective, and possibly linked,
	parameter at a pathname.
<P>
	@param	pathname	A parameter pathname.
	@return	The first String value from the effective parameter, or
		null if there is no value.
	@see	#Get_Linked(String)
	@see	#Get_One(Vector)
*/
public String Get_Linked_One (String pathname)
{return Get_One (Get_Linked (pathname));}

/**	Gets the value(s) for a Parameter.
<P>
	@param	parameter	The Parameter from which to get value(s).
	@return	A Vector of zero or more values. Each element of the
		Vector is either the String representation of a single
		value, or is itself a Vector of the same kind.
	@see	#Get(Value)
*/
public Vector Get
	(
	Parameter	parameter
	)
{
Value
	value = null;
if (parameter != null)
	{
	if ((DEBUG & DEBUG_GET) != 0)
		System.out.println ("Configuration.Get: parameter \""
			+ parameter.Path_Name () + "\"");
	try {value = parameter.Value ();}
	catch (PVL_Exception exception)
		{/* The parameter is not an Assignment. */}
	}
return Get (value);
}

/**	Gets one String value from the Get Vector.
<P>
	@param	parameter	The Parameter from which to get a value.
	@return	The first String value from the Vector, or null if there is
		no value.
	@see	#Get(Parameter)
	@see	#Get_One(Vector)
*/
public String Get_One
	(
	Parameter	parameter
	)
{return Get_One (Get (parameter));}

/**	Gets the value(s) for a Parameter Value.
<P>
	@param	value	The Value from which to get value(s).
	@return	A Vector of zero or more values. Each element of the
		Vector is either the String representation of a single
		value, or is itself a Vector of the same kind.
*/
public static Vector Get
	(
	Value	value
	)
{
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println ("Configuration.Get value: " + value);
Vector
	values = new Vector ();
if (value != null)
	{
	if (value.Is_Array ())
		{
		ListIterator
			list = value.listIterator ();
		while (list.hasNext ())
			{
			value = (Value)list.next ();
			if (value.Is_Array ())
				values.add (Configuration.Get (value));
			else
				{
				try {values.add (value.String_Data ());}
				catch (PVL_Exception exception)
					{/* Shouldn't happen. */}
				}
			}
		}
	else
		{
		try {values.add (value.String_Data ());}
		catch (PVL_Exception exception)
			{/* Shouldn't happen. */}
		}
	}
return values;
}

/**	Gets one String value from the Get Vector.
<P>
	@param	value	The Value from which to get a value.
	@return	The first String value from the Vector, or null if there is
		no value.
	@see	#Get(Value)
	@see	#Get_One(Vector)
*/
public String Get_One (Value value)
{return Get_One (Get (value));}

/**	Gets the first String value from a Vector of values.
<P>
	@param	values	The Vector to search.
	@return	The first String from the Vector, or null if none is found.
	@see	#Get_One(Vector, int)
*/
public static String Get_One
	(
	Vector	values
	)
{return Get_One (values, 0);}

/**	Gets the indexed String value from a Vector of values.
<P>
	The Vector is searched for the first non-Vector Object and its
	{@link Object#toString() toString} value is returned. Vector Objects
	are searched as they are encountered (i.e. depth-wise). Since the
	Vector is presumed to be the values Vector returned from a Get
	method, the first non-Vector Object should be the first String value
	of a Configuration parameter (nevertheless, all Objects should have
	a toString method).
<P>
	@param	values	The Vector to search.
	@param	index	The index of the element in the values.
	@return	The indexed String from the Vector, or null if none is found.
*/
public static String Get_One
	(
	Vector	values,
	int		index
	)
{
if (values != null &&
	values.size () > 0 &&
	values.size () > index)
	{
	Object
		value = values.elementAt (index);
	if (value instanceof Vector)
		return Get_One ((Vector)value, 0);
	return value.toString ();
	}
return null;
}

/**	Gets a numerical value.
<P>
	The first available parameter value found at the pathname is
	converted to a binary double value.
<P>
	@param	pathname		A parameter pathname.
	@param	default_value	The default value to use if the parameter can
		not be found.
	@return	The numerical value of the parameter, or the default value
		if the parameter can not be found or does not have a numerical
		value.
*/
public double Get_Number
	(
	String	pathname,
	double	default_value
	)
{
String
	value = Get_One (pathname);
if (value != null)
	{
	try {return Double.parseDouble (value);}
	catch (NumberFormatException exception) {}
	}
return default_value;
}

/** Tests if a parameter has a logically true value.
<P>
	The first available parameter value found at the pathname is
	examined for a logically true value. A paremeter is considered to
	have a logically true value if it is any one of (case ignored):
	"ENABLED", "TRUE", "YES", "ON", "Y", or "1".
<P>
	@param	pathname		A parameter pathname.
	@param	default_flag	The default value to use if the parameter can
		not be found.
	@return	true if the parameter has an enabled value; false otherwise.
		The default_flag value is returned if the parameter can not be found.
 */
public boolean Enabled
	(
	String	pathname,
	boolean	default_flag
	)
{
String
	value = Get_One (pathname);
if (value == null)
	return default_flag;
if (value.equalsIgnoreCase ("ENABLED") ||
	value.equalsIgnoreCase ("TRUE") ||
	value.equalsIgnoreCase ("YES") ||
	value.equalsIgnoreCase ("ON") ||
	value.equalsIgnoreCase ("Y") ||
	value.equalsIgnoreCase ("1"))
	return true;
return false;
}

/**	Gets the Configuration parameters for a group.
<P>
	All parameters for a named group, plus all default parameters for
	the group, are collected into a new Configuration that will have the
	name of the group. Default parameters are all Assignment parameters
	in all parents of the group parameter. These parameters are {@link
	Configuration#Set_Conditionally(String, Object) Set_Conditionally}
	in the new configuration such that parameters closest to the group
	(a parent is closer than a grandparent) have precedence when there
	are parameters with the same name.
<P>
	If an Assignment parameter with the group name is found instead of
	an Aggregate parameter, its value is taken to be the name of the
	Aggregate parameter name to use. Such a parameter is a "link" to the
	actual group parameter, which gives the group parameter an "alias"
	name. This allows configuration parameter groups to be known by
	different names. Link parameters may refer to other link parameters.
	If the extracted configuration is an alias it will be given an
	{@link Configuration#ALIAS ALIAS} parameter having the value that is
	the name of the alias group.
<P>
	@param	group	The name of the configuration group to extract.
	@return	A Configuration with all of the group's parameters,
		or null if the group could not be found.
	@throws Configuration_Exception	If the Parameter group could not be
		assembled into a new Configuration.
*/
public Configuration Group
	(
	String		group
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (">>> Configuration.Group: " + group);
Configuration
	group_configuration = null;
Parameter
	parameter = null;
String
	alias = group;
while ((parameter = Find (alias, Case_Sensitive, parameter)) != null &&
	  ! parameter.Is_Aggregate ())
	{
	if (parameter.Is_Assignment ())
		{
		try
			{
			//	Follow name aliases (links).
			if (parameter.Value () != null &&
				! (alias = parameter.Value ().String_Data ()).equals ("") &&
				! alias.equals (group))
				{
				//	Find the new name from the top.
				parameter = null;
				if ((DEBUG & DEBUG_UTILITY) != 0)
					System.out.println ("    alias: " + alias);
				}
			}
		catch (PVL_Exception exception)
			{/* Shouldn't happen */ return null;}
		}
	}
if (parameter != null)
	{
	group_configuration = new Configuration (parameter);
	if (! alias.equals (group))
		{
		group_configuration.Set (Configuration.ALIAS, alias);
		group_configuration.Name (group);
		}
	//	Bring down all the Assignments from the parents.
	Parameter
		parent = parameter;
	while ((parent = parent.Parent ()) != null)
		{
		ListIterator
			list = parent.listIterator ();
		while (list.hasNext ())
			{
			parameter = (Parameter)list.next ();
			try
				{
				if (parameter.Is_Assignment ())
					group_configuration.Set_Conditionally
						(new Parameter (parameter));
				}
			catch (PVL_Exception exception)
				{
				throw new Configuration_Exception
					(
					ID + NL
					+ "Unable to copy group parameter \""
						+ parameter.Name () + "\"."
					+ exception_message (exception),
					exception
					);
				}
			}
		}
	group_configuration.Coalesce ();
	}
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println ("<<< Configuration.Group: "
		+ (! (group_configuration == null)));
return group_configuration;
}

/*-----------------------------------------------------------------------------
*/
/**	Sets a parameter at a specified pathname location to the value
	of an Object.
<P>
	The Object may be of any type acceptable for use in creating a
	Parameter Value. This includes Vectors of Objects which will set
	the parameter to the corresponding Value array. A null value will
	result in an empty, Token, parameter.
<P>
	When the value of an existing parameter is not to be replaced, the
	availability of an {@link #Effective_Parameter(String)
	Effective_Parameter} is sufficient to prevent the creation of a new
	parameter. However, when replace is true, a parameter will be
	created at the specific pathname if it does not already exist.
	<B>Note</B>: If an existing parameter is found for the pathname,
	whether specific or effective, that does not have a value, its
	always assigned the new value.
<P>
	Only parameters assigned a value - not parameter groups - may have
	their value set. If the parameter does not exist a new parameter is
	created and assigned the value.
<P>
	@param	pathname	The pathname of the parameter whose value is to
		be set.
	@param	value	The value that will be assigned to the parameter.
	@param	replace	If true, replace the values of existing parameters
		with the new value; if false, do not replace existing values.
	@return	true if a previous value was replaced, or false if the
		parameter is being set for the first time.
	@throws	Configuration_Exception	If a new parameter could not be
		created (possibly because the pathname is null), or the value
		could not be set.
	@see	Parameter#Value(Object)
*/
public synchronized boolean Set
	(
	String	pathname,
	Object	value,
	boolean	replace
	)
	throws Configuration_Exception
{
if (pathname == null)
	throw new Configuration_Exception
		(
		ID + NL
		+ "Unable to Set a parameter with a null pathname."
		);
if ((DEBUG & DEBUG_SET) != 0)
	System.out.println
		(">>> Set: " + pathname + " to " + value
		+ " (replace = " + replace + ")");
//	Use effective parameters when not replacing (replace == specific).
Parameter
	parameter = Parameter (pathname, replace);
if ((DEBUG & DEBUG_SET) != 0)
	System.out.println
		("Set: " + pathname
		+ ((parameter == null) ? " not found." : " found."));
Value
	old_value = null;
if (parameter == null)
	{
	//	Create a new (initially empty) parameter.
	parameter = group_parameter (pathname);
	if ((DEBUG & DEBUG_SET) != 0)
		System.out.println
			("Set: new parameter - " + parameter.Path_Name ());
	}
else
	{
	//	Get the previous value.
	try {old_value = parameter.Value ();}
	catch (PVL_Exception exception) {/* Shouldn't happen. */}
	}
if (replace || old_value == null)
	{
	//	Replace the parameter's value.
	if ((DEBUG & DEBUG_SET) != 0)
		System.out.println
			("Set: replacing " + old_value + " with " + value);
	try {parameter.Value (value);}
	catch (PVL_Exception exception)
		{
		throw new Configuration_Exception
			(
			ID + NL
			+ "Unable to Set parameter \"" + pathname + "\""
				+ " to value \"" + value + "\"."
			+ exception_message (exception),
			exception
			);
		}
	}
if ((DEBUG & DEBUG_SET) != 0)
	System.out.println
		("<<< Set: Previous value - "
		+ ((old_value == null) ? "false" : "true"));
return (old_value == null) ? false : true;
}

/**	Sets the value of a parameter at a pathname.
<P>
	A new parameter is created with the value if a parameter with the
	pathname does not exist. Otherwise the existing value is replaced.
	This is the same as using the {@link #Set(String, Object, boolean)
	Set} method with a replace argument of true.
<P>
	@param	pathname	The pathname of the parameter whose value is to
		be set.
	@param	value	The value that will be assigned to the parameter.
	@return	true if a previous value was replaced, or false if the
		parameter is being set for the first time.
	@see	#Set(String, Object, boolean)
*/
public boolean Set
	(
	String	pathname,
	Object	value
	)
	throws Configuration_Exception
{return Set (pathname, value, true);}

/**	Sets the value of a new parameter at a pathname.
<P>
	A new parameter is created with the value if a parameter with the
	pathname does not exist. Otherwise the existing value is left
	unchanged. This is the same as using the {@link #Set(String, Object,
	boolean) Set} method with a replace argument of false.
<P>
	@param	pathname	The name of the new parameter.
	@param	value	The value that will be assigned to the parameter.
	@return	true if a previous value was replaced, or false if the
		parameter is being set for the first time.
	@throws	Configuration_Exception	If there was a problem setting any
		parameter value.
	@see	#Set(String, Object, boolean)
*/
public boolean Set_Conditionally
	(
	String	pathname,
	Object	value
	)
	throws Configuration_Exception
{return Set (pathname, value, false);}

/**	Sets all of the entries of an array of String arrays as
	Configuration parameters.
<P>
	Each entry of the primary array is taken to be an array of Strings.
	The first String is a parameter pathname and all remaining Strings
	in the array are values for the parameter. A parameter pathname
	without any following parameter value(s) will be skipped.
<P>
	@param	parameter_table	The table of parameter names and
		values.
	@param	replace	If true, replace the values of existing parameters
		with the new value; if false, do not replace existing values.
	@throws	Configuration_Exception	If there was a problem setting any
		parameter value.
	@see	#Set(String, Object, boolean)
*/
public void Set
	(
	String[][]	parameter_table,
	boolean		replace
	)
	throws Configuration_Exception
{
Vector
	values = new Vector ();
for (int element = 0;
		 element < parameter_table.length;
		 element++)
	{
	values.clear ();
	int
		value;
	for (value = 1;
		 value < parameter_table[element].length;
		 value++)
		values.add (parameter_table[element][value]);
	if (value > 1)
		{
		try
			{
			if (values.size () == 1)
				//	Single value (don't create an Array Value).
				Set
					(
					parameter_table[element][0],
					(String)values.elementAt (0),
					replace
					);
			else
				Set
					(
					parameter_table[element][0],
					values,
					replace
					);
			}
		catch (Configuration_Exception exception)
			{
			throw new Configuration_Exception
				(
				ID + NL
				+ "During Set of \"" + parameter_table[element][0]
					+ "\" Strings array entry " + element + "."
				+ exception_message (exception),
				exception
				);
			}
		}
	}
}

/**	Sets all of the entries of an array of String arrays.
<P>
	An existing parameter has its value replaced, otherwise a new
	parameter is created 
<P>
	@param	parameter_table	The table of parameter names and
		values.
	@throws	Configuration_Exception	If there was a problem setting any
		parameter value.
	@see	#Set(String[][], boolean)
	@see	#Set(String, Object)
*/
public void Set
	(
	String[][]	parameter_table
	)
	throws Configuration_Exception
{Set (parameter_table, false);}

/**	Conditionally sets all of the entries of an array of String arrays.
<P>
	New parameters are created; existing parameters are unchanged.
<P>
	@param	parameter_table	The table of parameter names and
		values.
	@throws	Configuration_Exception	If there was a problem setting any
		parameter value.
	@see	#Set(String[][], boolean)
	@see	#Set_Conditionally(String, Object)
*/
public void Set_Conditionally
	(
	String[][]	parameter_table
	)
	throws Configuration_Exception
{Set (parameter_table, false);}

/**	Sets parameters in the Configuration to correspond with the
	contents of another parameter.
<P>
	For an Assignment parameter, a parameter in the Configuration at
	the same pathname location is assigned the same parameter value.
	For a parameter group, all of its members are recursivley set in
	the Configuration.
<P>
	The {@link Parameter#Path_Name() pathname} of each parameter that
	is set is made to appear to be rooted at the Configuration root.
	Thus the location of each parameter set in the Configuration will
	be at the same relative location of the source parameter in <I>its
	own hierarchy</I> according to its own pathname. Referencing a
	source parameter nested somewhere down in a hierachy will then
	produce a Configuration parameter that is nested down to the same
	relative location. However, when the intention is to root
	the Configuration parameters relative to the source parameter being
	at the root itself, then the source parameter should be copied
	before being provided to this method:
	<P><BLOCKQUOTE>
		<CODE>Set (new Parameter (parameter), replace);</CODE>
	</BLOCKQUOTE><P>
	<B>Note</B>: Freestanding source parameters (that are not located
	in a group hierarchy) set Configuration parameters located at the
	root.
<P>
	@param	parameter	The source Parameter for setting in the
		Configuration.
	@param	replace	If true, replace the values of existing parameters
		with the new value; if false, do not replace existing values.
	@throws	Configuration_Exception	If there was a problem setting
		any parameter in the Configuration.
	@see	#Set(String, Object, boolean)
*/
public void Set
	(
	Parameter	parameter,
	boolean		replace
	)
	throws Configuration_Exception
{
if (parameter == null)
	return;
if ((DEBUG & DEBUG_SET) != 0)
	System.out.println (">>> Set:" + NL
		+ "    parameter: " + parameter.Path_Name () + NL
		+ "        class: " + parameter.Classification_Name () + NL
		+ "      replace: " + replace);
String
	pathname;
if (parameter.Is_Aggregate ())
	{
	ListIterator
		list = parameter.listIterator ();
	while (list.hasNext ())
		{
		Parameter
			new_parameter = (Parameter)list.next ();
		if ((DEBUG & DEBUG_SET) != 0)
			System.out.println
				("    Set Aggregate entry " + new_parameter.Path_Name ());
		pathname = Path_Name_to_Pathname (new_parameter.Path_Name ());
		if (new_parameter.Is_Aggregate () && new_parameter.Is_Empty ())
			group_parameter (pathname)
				.Classification (new_parameter.Classification ());
		else
			{
			try {Set (new_parameter, replace);}
			catch (Configuration_Exception exception)
				{
				throw new Configuration_Exception
					(
					ID + NL
					+ "During Set of parameter group \""
					+ parameter.Name () + '"' + NL
					+ "  at entry " + list.previousIndex () + "."
					+ exception_message (exception),
					exception
					);
				}
			}
		}
	}
else
	{
	pathname = Path_Name_to_Pathname (parameter.Path_Name ());
	if (pathname.length () == 0)
		pathname = parameter.Path_Name ();
	//	Use specific pathname.
	if (replace || Parameter (pathname, true) == null)
		{
		Value
			value = null;
		try {value = parameter.Value ();}
		catch (PVL_Exception exception) {}
		Set (pathname, value);
		//	Just let Set throw any exception.
		}
	}
if ((DEBUG & DEBUG_SET) != 0)
	System.out.println ("<<< Set parameter " + parameter.Path_Name ());
}

/**	Sets parameters in the Configuration to correspond with the
	contents of another parameter.
<P>
	An existing Configuration parameter has its value replaced,
	otherwise a new parameter is created 
<P>
	@param	parameter	The source Parameter for setting in the
		Configuration.
	@throws	Configuration_Exception	If there was a problem setting any
		parameter value.
	@see	#Set(Parameter, boolean)
*/
public void Set
	(
	Parameter	parameter
	)
	throws Configuration_Exception
{Set (parameter, true);}

/**	Conditionally sets parameters in the Configuration to correspond
	with the contents of another parameter.
<P>
	New parameters are created; existing parameters are unchanged.
<P>
	@param	parameter	The source Parameter for setting in the
		Configuration.
	@throws	Configuration_Exception	If there was a problem setting any
		parameter value.
	@see	#Set(Parameter, boolean)
	@see	#Set_Conditionally(String, Object)
*/
public void Set_Conditionally
	(
	Parameter	parameter
	)
	throws Configuration_Exception
{Set (parameter, false);}

/**	Sets all occurances of a parameter with the specified name to have
	the specified value.
<P>
	Only existing parameters have their value reset. If there are no
	parameters with the specified name, no new parameter is created.
<P>
	@param	name	The parameter name to find. This should be a simple
		name, not a pathname. If a pathname is specified, then only
		parameters at the pathname will be modified; an absolute
		pathname can only apply to a single parameter, but a relative
		pathname may apply to more than one.
	@param	value	The Object to be assigned as the value of each
		parameter found.
	@return	true if at least one parameter was found, false otherwise.
	@throws	Configuration_Exception	If there was a problem replacing a
		parameter's value.
*/
public synchronized boolean Set_All
	(
	String		name,
	Object		value
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_SET) != 0)
	System.out.println (">>> Set_All (\"" + name + "\", " + value + ")");
boolean
	found = false;
Parameter
	pattern = new Parameter (name);
Selector
	selection = new Selection ()
		.Name (true)
		.Specific (Case_Sensitive);
Parameter
	parameter = null;
while ((parameter = Find (pattern, selection, parameter)) != null)
	{
	if (parameter.Is_Aggregate ())
		continue;
	found = true;
	try
		{
		//	Replace the parameter's value.
		if ((DEBUG & DEBUG_SET) != 0)
			System.out.println
				("Set_All: replacing "
				+ parameter.Path_Name () + ":" + parameter.Value ());
		parameter.Value (value);
		}
	catch (PVL_Exception exception)
		{
		throw new Configuration_Exception
			(
			ID + NL
			+ "Unable to Set parameter \"" + parameter.Path_Name () + "\""
				+ " to value \"" + value + "\"."
			+ exception_message (exception),
			exception
			);
		}
	}
return found;
}

/**	Returns the group Parameter at the pathname.
<P>
	This Parameter is created (as an empty Parameter) if it doesn't
	exist; which means that any and all of its parent Parameters must
	exist or will be created, too. If the pathname is relative (i.e.
	does not begin with a {@link Parameter#Path_Delimiter() path
	delimiter}) and the first segment of the pathname does not exist,
	then it will be created at the root of the Configuration.
	<B>Note</B> The Parameter returned will not be a group parameter if,
	and only if, a new parameter was created.
<P>
	@param	pathname	The pathname to a group parameter.
	@return	The Parameter at the pathname location.
	@throws	Configuration_Exception	If there is a problem creating any
		segment of the pathname, including if the pathname is null.
*/
private synchronized Parameter group_parameter
	(
	String	pathname
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_SET) != 0)
	System.out.println
		("... group_parameter: " + pathname);
if (pathname == null)
	throw new Configuration_Exception
		(
		ID + NL
		+ "Null pathname for group_parameter!"
		);
//	Find only group parameters with the specified name.
Selector
	selection = new Selection ()
		.Name (true)
		.Specific (Case_Sensitive)
		.And (true)
		.Classification (true);
Parameter
	pattern;
try {pattern = new Parameter (pathname).Add (new Parameter ());}
catch (PVL_Exception exception)
	{
	//	This shouldn't happen.
	throw new Configuration_Exception
		(
		ID + NL
		+ "Unable to construct group parameter pattern parameter!"
		+ exception_message (exception),
		exception
		);
	}
Parameter
	parameter = Find (pattern, selection);
if ((DEBUG & DEBUG_SET) != 0 && parameter != null)
	System.out.println
		("    group_parameter: Found " + pathname);
if (parameter == null)
	{
	//	No such parameter. Create a new one.
	parameter = new Parameter (Basename (pathname));
	try
		{
		//	Get the parent Parameter in which to put it.
		if ((pathname = Parent_Pathname (pathname)) != null)
			{
			if ((DEBUG & DEBUG_SET) != 0)
				System.out.println
					("    group_parameter: Add "
					+ parameter.Path_Name () + " to " + pathname);
			group_parameter (pathname).Add (parameter);
			}
		else
			{
			//	There is no parent (the pathname is at the root).
			if ((DEBUG & DEBUG_SET) != 0)
				System.out.println
					("    group_parameter: Add "
					+ parameter.Path_Name () + " at root.");
			Add (parameter);
			}
		}
	catch (PVL_Exception exception)
		{
		throw new Configuration_Exception
			(
			ID + NL
			+ "Unable to add a new \"" + pathname + "\" parameter."
			+ exception_message (exception),
			exception
			);
		}
	}
return parameter;
}

/*-----------------------------------------------------------------------------
*/
/**	Removes the parameter(s) at a pathname.
<P>
	@param	pathname	A parameter pathname.
	@param	specific	If true, remove only the parameter at the
		specific pathname, otherwise remove all effective parameters
		for the pathname.
	@return	true if a previous value was removed, or false if there
		is no parameter at the pathname.
*/
public synchronized boolean Remove
	(
	String	pathname,
	boolean	specific
	)
{
Parameter
	parameter;
boolean
	removed = false;
if (specific)
	{
	if ((parameter = Specific_Parameter (pathname)) != null)
		{
		parameter.Parent ().Remove (parameter);
		return true;
		}
	}
else
	{
	while ((parameter = Effective_Parameter (pathname)) != null)
		{
		parameter.Parent ().Remove (parameter);
		removed = true;
		}
	}
return removed;
}

/**	Removes the all of the effective parameters for the pathname.
<P>
	This is the same as using the {@link #Remove(String, boolean)
	Remove} method with the specific argument being false.
<P>
	@param	pathname	A parameter pathname.
	@return	true	If a previous value was removed, or false if there
		were no effective parameters for the pathname.
*/
public boolean Remove
	(
	String	pathname
	)
{return Remove (pathname, false);}

/**	Removes the parameter at the specific pathname.
<P>
	This is the same as using the {@link #Remove(String, boolean)
	Remove} method with the specific argument being true.
<P>
	@param	pathname	A parameter pathname.
	@return	true	If a previous value was removed, or false if there
		was no parameter at the pathname.
*/
public boolean Remove_Specific
	(
	String	pathname
	)
{return Remove (pathname, true);}

/**	Removes a group of parameters at a pathname.
<P>
	The group of parameters at the pathname is removed.
<P>
	@param	pathname	A group pathname.
	@return	true	If a group was removed, or false if there was no
		group at the pathname.
*/
public boolean Remove_Group
	(
	String	pathname
	)
{
boolean
	removed = false;
Parameter
	parameter;
if ((parameter = Parameter_at (pathname)) != null &&
	parameter.Is_Aggregate ())
	{
	parameter.Parent ().Remove (parameter);
	removed = true;
	}
return removed;
}

/*-----------------------------------------------------------------------------
*/
/**	Gets the name of the source from which the Configuration parameters
	have been set.
<P>
	@return	The source name String for the Configuration. This will be a
		source file pathname or URL if the Configuration was constructed
		from such a source. This will be {@link #DEFAULTS DEFAULTS} if
		the Configuration is based soley on the internal default
		parameters. It will be the {@link Parser#CONTAINER_NAME} if the
		Configuration was constructed from an Assignment Parameter
		(including an empty Parameter) rather than an Aggregate. If the
		Configuration was constructed from an Aggregate Parameter it
		will be the name of the Aggregate Parameter.
*/
public String Source ()
{return Config_Source;}

/**	Gets the name of the file from which the Configuration parameters
	have been set.
<p>
	@deprecated	Replaced by {@link #Source()}
*/
public String Filename ()
{return Source ();}

/**	Resets the Configuration from the contents of the source.
<P>
	@param	source	The name of the configuration source.
	@return	This Configuration.
	@throws	Configuration_Exception	If the source can not be read or
		successfully parsed for parameters, or there was a problem
		coalescing the parameters (probably due to a duplicate
		parameter name).
	@see	#Configure(String)
*/
public Configuration Source
	(
	String	source
	)
	throws Configuration_Exception
{
Configure (source);
return this;
}

/**	Resets the Configuration from the contents of the source file.
<P>
	@param	filename	The name of the configuration file.
	@deprecated	Replaced by {@link #Source(String)}
*/
public Configuration Filename
	(
	String	filename
	)
	throws Configuration_Exception
{return Source (filename);}

/**	Sets the default source name.
<P>
	@param	source	The name of the new default source. If null, the
		{@link #DEFAULTS DEFAULTS} name will be used.
*/
public static void Default_Source
	(
	String source
	)
{
if ((Default_Source = source) == null)
	 Default_Source = DEFAULTS;
}

/**	Sets the default filename.
<P>
	@param	filename	The new default filename. If null, the {@link
		#DEFAULTS DEFAULTS} name will be used.
	@deprecated	Replaced by {@link #Default_Source(String)}
*/
public static void Default_Filename
	(
	String filename
	)
{Default_Source (filename);}

/**	Gets the default source name.
<P>
	@return	The default source name.
	@see	#Default_Source(String)
*/
public static String Default_Source ()
{return Default_Source;}

/**	Gets the default source name.
<P>
	@return	The default source name.
	@deprecated	Replaced by {@link #Default_Source()}
*/
public static String Default_Filename ()
{return Default_Source ();}

/**	Sets the class relative to which a configuration file may be sought.
<p>
	@param	related_class	The Class related to the configuration file
		to be loaded.
	@see	#Configure(String)
*/
public static void Relative_to_Class
	(
	Class	related_class
	)
{Relative_to_Class = related_class;}

/**	Gets the class relative to which a configuration file may be sought.
<p>
	@return	The Class related to the configuration file to be loaded.
	@see	#Relative_to_Class(Class)
*/
public static Class Relative_to_Class ()
{return Relative_to_Class;}

/**	Ensure that this Configuration is an Aggregate.
<P>
	If this Configuration is an Aggregate, nothing is done.
<P>
	If this Configuration is not an Aggregate, it is made into one. If
	this Configuration has the UNKNOWN classification (it is empty) it's
	Classification is simply changed to GROUP. Otherwise a copy of the
	Parameter that is currently this Configuration is added to this
	Configuration, thus making this Configuration an Aggregate while
	retaining the current Parameter. Then the name of this Configuration
	is set to {@link Parser#CONTAINER_NAME}.
*/
private void Ensure_Aggregate_Configuration ()
{
if (! Is_Aggregate ())
	{
	if (Is_Unknown ())
		Classification (GROUP);
	else
		{
		try {Add (new Parameter (this));}
		catch (PVL_Exception exception) {/* This is a good Parameter */}
		}
	Name (Parser.CONTAINER_NAME);
	}
}

/*-----------------------------------------------------------------------------
*/
/**	Sets the case sensitive mode for finding parameter names.
	It is initially disabled by default.
<P>
	@param	enable	If true, case sensitive matching is used.
	@return	The previous case sensitive mode;
*/
public boolean Case_Sensitive
	(
	boolean	enable
	)
{
boolean
	previous_state = Case_Sensitive;
Case_Sensitive = enable;
return previous_state;
}

/**	Gets the case sensitive mode for finding parameter names.
<P>
	@return	The case sensitive mode: true if case sensitive matching is
		enabled.
	@see	#Case_Sensitive(boolean)
*/
public boolean Case_Sensitive ()
{return Case_Sensitive;}

/**	Enables or disables automatic inclusion of {@link #Defaults}
	parameters.
<P>
	<B>N.B.</B>: This method controls the automatic inclusion of {@link
	#Defaults} parameters for subsequent Configure operations. The
	{@link #Defaults_Default default automatic defaults} state is
	applied to the construction of new Configuration objects.
<P>
	@param	enable	true if Defaults parameters are to be automatically
		included; false otherwise.
	@return	The previous automatic Defaults state.
*/
public boolean Defaults
	(
	boolean	enable
	)
{
boolean
	state = Include_Defaults;
Include_Defaults = enable;
return state;
}

/*==============================================================================
	Methods
*/
/**	Provides a String description of a Configuration formatted as a
	list of the parameters and their values.
<P>
	The description is suitable for saving to a file for later input.
<P>
	@return	The String listing all of the Configuration parameters.
*/
public String Description ()
{
StringWriter
	description = new StringWriter ();
String
	name = Name ();
description.write (Parser.CROSSHATCH);
description.write (" " + name + NL);
if (Config_Source != null)
	description.write (Parser.CROSSHATCH + " From: " + Config_Source + NL);
Lister
	lister = new Lister (description).Strict (false);

//	Prevent the top level from being written.
Name (Parser.CONTAINER_NAME);
try {lister.Write (this);}
catch (Exception exception)
	{
	Name (name);	//	Restore the original name.
	return exception.getMessage ();
	}
Name (name);	//	Restore the original name.

if (lister.Warning () != null)
	{
	String_Buffer
		warning = new String_Buffer (lister.Warning ().getMessage ());
	warning.escape_to_special ();
	description.getBuffer ()
		.append (NL + "Warning:" + NL)
		.append (warning.toString ());
	}
return description.toString ();
}

/*-----------------------------------------------------------------------------
*/
/**	Loads a Configuration from a configuration file.
<P>
	If the source String is null or empty, the {@link #Default_Source()
	Default_Source} String is used. If this is null the {@link #DEFAULTS}
	name is used.
<P>
	If the source String equals the {@link #DEFAULTS} name the
	Configuration is {@link #Configure(InputStream) loaded from a null
	stream}.
<P>
	If a URL can be constructred from the source String the Configuration
	is {@link #Configure(URL) loaded from the URL}.
<P>
	An attempt is made to open a file input stream use the source String
	as the file's pathname. If this fails and the source String is a
	simple filename - it has no directory components - the user's home
	directory pathname is prepended and another attempt is made to open a
	file input stream (unless the simple filename is for the same file in
	the user's home directory). If a stream is opened it is used to
	{@link #Configure(InputStream) load} the Configuration.
<P>
	The source String is then treated as a {@link
	ClassLoader#getSystemResource(String) system resource location} in
	the JVM classpath. This includes finding configuration files located
	in jar files. If this fails, a {@link #Relative_to_Class(Class)
	relative-to Class} has been specified and the source String is a
	simple filename then the class's package pathname, if it can be
	determined, is prepended to the source name and another attempt is
	made to locate the system resource. If a system resource location is
	found the Configuration is {@link #Configure(URL) loaded from the
	URL}.
<P>
	When a input source is found the {@link #Source() Source} is set to
	the name of the source. If the source is a local file the absolute
	pathname will be used. If the source is a URL, including a system
	resource location URL, the URL will be used.
<P>
	@param	source	A URL specification, pathname of a disk file or jar
		resource, or the special {@link #DEFAULTS DEFAULTS} name. If
		null, the {@link #Default_Source() Default_Source} will be
		used.
	@throws	Configuration_Exception	If the source does not refer to a
		source that can be found, read or successfully parsed for
		parameters, or there was a problem coalescing the parameters
		(probably due to a duplicate parameter name).
	@throws	IllegalArgumentException	If no configuration source can be
		found for the source name.
	@see	#Configure(URL)
	@see	#Configure(InputStream)
	@see	#Source()
*/
public synchronized void Configure
	(
	String	source
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println (">>> Configure (source: " + source + ")");

if (source == null ||
	source.length () == 0)
	source = Default_Source;
if (source == null ||
	source.length () == 0)
	source = DEFAULTS;

String
	config_source = null;
FileInputStream
	stream = null;
if (! source.equals (DEFAULTS))
	{
	//	Try to construct a URL from the source String.
	URL
		url = null;
	try 
		{
		url = new URL (source);
		Configure (url);
		return;
		}
	catch (MalformedURLException exception)
		{/* Nope. Fall back... */}

	//	Try to find a local file.
	File
		file = new File (source);
	config_source = file.getAbsolutePath ();
	if ((DEBUG & DEBUG_CONFIGURE) != 0)
		System.out.println ("Configure: Trying " + config_source);
	try {stream = new FileInputStream (file);}
	catch (FileNotFoundException exception)
		{
		/* Nope. Fall back... */
		if ((DEBUG & DEBUG_CONFIGURE) != 0)
			System.out.println
				("Configure: No configuration file " + config_source);
		}
	if (stream == null &&
		file.getName ().equals (source))
		{
		/*	Just a simple filename. 
			Try looking for it in the user's home directory.
		*/
		String
			pathname =
				System.getProperty ("user.home") + File.separator + source;
		if (! pathname.equals (config_source))
			{
			config_source = pathname;
			if ((DEBUG & DEBUG_CONFIGURE) != 0)
				System.out.println ("Configure: Trying " + pathname);
			try {stream = new FileInputStream (pathname);}
			catch (FileNotFoundException exception)
				{
				/* Nope. Fall back... */
				if ((DEBUG & DEBUG_CONFIGURE) != 0)
					System.out.println
						("Configure: No configuration file " + pathname);
				}
			}
		}
	if (stream == null)
		{
		/*	Try for a URL from a System Resource at the source location.

			This can find a file inside a jar file.
		*/
		url = ClassLoader.getSystemResource (source);
		if (url == null &&
			Relative_to_Class != null &&
			file.getName ().equals (source))
			{
			//	Try for a class relative location.
			try {url = ClassLoader.getSystemResource
					(Relative_to_Class.getPackage ().getName ()
						.replace ('.', File.separatorChar)
					+ File.separator + source);}
			catch (NullPointerException except) {}
			}
		if (url == null)
			{
			/*	If url is null, then we were unable even to use the source
				string to get a System Resource. At this point, we are out of
				reasonable options and must throw an exception.
			*/
			throw new IllegalArgumentException
				(
				ID + NL
				+ "Configure can't find source file \"" + source + "\"."
				);
			}
		Configure (url);
		return;
		}
	}
if (config_source != null)
	Config_Source = config_source;

Configure (stream);
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println ("<<< Configure (source: " + source + ")");
}

/**	Loads a Configuration from a URL.
<P>
	The URL is used to open an InputStream as a source of configuration
	parameters. If the URL is null then a null stream will be used. If a
	stream is successfully opened from the URL, then the Configuration's
	{@link #Source() Source} will be set to the URL String. The
	Configuration is {@link #Configure(InputStream) loaded from the
	InputStream}.
<P>
	@param	url	The URL source of configuration parameters. May be null.
	@throws	Configuration_Exception	If an InputStream can not be opened
		from the URL, the stream could not be successfully parsed for
		parameters, or there was a problem coalescing the parameters
		(probably due to a duplicate parameter name).
	@see	#Configure(InputStream)
*/
public void Configure
	(
	URL		url
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println (">>> Configure (URL: " + url + ")");
InputStream
	stream = null;
if (url != null)
	{
	try {stream = url.openStream ();}
	catch (IOException exception)
		{
		if ((DEBUG & DEBUG_CONFIGURE) != 0)
			System.out.println ("Unable to open a stream from the URL "
				+ url + NL
				+ exception.getMessage ());
		throw new Configuration_Exception
			(
			ID + NL
			+ "Configure can't open URL -" + NL
			+ url
			+ exception_message (exception),
			exception
			);
		}
	Config_Source = url.toString ();
	}
Configure (stream);
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println ("<<< Configure (URL: " + url + ")");
}

/**	Loads a Configuration from an InputStream.
<P>
	The stream is parsed for configuration parameters and their values
	(parameter = value).
<P>
	This method will first {@link Parameter#Remove_All() Remove_All}
	current parameters from the Configuration. The InputStream is used
	to construct a Parser from which all available parameters are {@link
	Parameter#Add(Parser) Add}ed into this Configuration. If automatic
	{@link #Defaults(boolean) Defaults} are enabled the internal
	parameter {@link #Defaults Defaults} are then {@link
	#Set_Conditionally(Parameter) Set_Conditionally}. If automatic file
	{@link #Include(boolean) Include}s is enabled any {@link #INCLUDE}
	parameters are found and the parameters from the files they
	reference are {@link #Include() Include}d. Finally all parameters
	are {@link #Coalesce() Coalesce}d together.
<P>
	If, however, the InputStream is null, then only the Defaults will be
	used. 
	 The Configuration's {@link #Source() Source} will be
	set to the special {@link #DEFAULTS DEFAULTS} name.
<P>
	If this Configuration's Source is not null, then the {@link
	Parameter#Name() Name} will be set to the same String; otherwise it
	will simply be set to "InputStream."
<P>
	@param	stream	The InputStream source of configuration parameters.
	@throws	Configuration_Exception	If the stream can not be read or
		successfully parsed for parameters, or there was a problem
		coalescing the parameters (probably due to a duplicate
		parameter name).
	@see	Parser#Parser(InputStream)
*/
public synchronized void Configure
	(
	InputStream	stream
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println (">>> Configure (InputStream)");
//	Clear all current parameters.
Remove_All ();

if (stream != null)
	{
	try
		{
		//	Add all parameters parsed from the stream.
		Add ((Parser)(new Parser (stream)
			.Strict (false)
			.Crosshatch_Comments (true)
			.Filter_Input (false)));
		}
	catch (PVL_Exception exception)
		{
		if ((DEBUG & DEBUG_CONFIGURE) != 0)
			System.out.println
				("Configure can't Add Parameters from InputStream Parser -" + NL
				+ exception.Message ());
		if (exception.Message ().equals (PVL_Exception.FILE_IO))
			throw new Configuration_Exception
				(
				ID + NL
				+ "Configure can't read Parameters from InputStream Parser."
				+ exception_message (exception),
				exception
				);
		else
			throw new Configuration_Exception
				(
				ID + NL
				+ "Configure couldn't get Parameters from InputStream Parser."
				+ exception_message (exception),
				exception
				);
		}
	}
else
	{
	//	DEFAULTS
	if ((DEBUG & DEBUG_CONFIGURE) != 0)
		System.out.println ("Configure: Using " + DEFAULTS);
	Config_Source = DEFAULTS;
	}
if (Config_Source != null)
	{
	/*	Set the Configuration name to the name of its source file.

		Only the basename of Config_Source is used because the full
		pathname is likely to contain delimiters that are the same as
		the Parameter reference path delimiter character. This will
		confuse operations that use parameter pathname references which
		are assembled from parameter names.
	*/
	File
		filename = new File (Config_Source);
	Name (filename.getName ());
	}
else
	Name ("InputStream");

//	Ensure that an empty or single parameter configuration is a group.
Ensure_Aggregate_Configuration ();

if (Include_Defaults)
	//	Make sure all of the default parameters have been set.
	Set_Conditionally (Defaults);

if (Include_Includes)
	//	Include all @include files.
	Include ();

//	Coalesce all the parameters.
Coalesce ();

if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println ("<<< Configure (InputStream)");
}

/**	Adds environment variables to the Configuration.
<P>
	All environment variables that can be obtained from the runtime
	environment are added to the {@link #ENVIRONMENT} group. If the
	group doesn't exist, it is appended to the end of the Configuration.
	Each environment variable is used to conditionally set a parameter
	in the Environment group; i.e. any existing parameters with the same
	name as an environment variable do not have their values changed.
	The value of an environment variable is always a single String
	regardless of embedded spaces or commas. 
<P>
	@return	This Configuration.
	@see	Host#Environment()
*/
public Configuration Add_Environment ()
	throws Configuration_Exception
{
Parameter
	environment_group = null;
try {environment_group = group_parameter (ENVIRONMENT);}
catch (Configuration_Exception exception)
	{
	throw new Configuration_Exception
		(
		ID + NL
		+ "Unable to add Environment group."
		+ exception_message (exception),
		exception
		);
	}
if (! environment_group.Is_Aggregate ())
	environment_group.Classification (Parameter.GROUP);

Properties
	environment_variables = Host.Environment ();
Enumeration
	variables = environment_variables.propertyNames ();
while (variables.hasMoreElements ())
	{
	String
		name = (String)variables.nextElement ();
	try
		{
		Set_Conditionally (ENVIRONMENT + Path_Delimiter () + name,
			environment_variables.getProperty (name));
		}
	catch (Configuration_Exception exception)
		{
		throw new Configuration_Exception
			(
			ID + NL
			+ "Unable to add Environment variable:" + NL
			+ name + " = " + environment_variables.getProperty (name)
			+ exception_message (exception),
			exception
			);
		}
	}
return this;
}

/**	Enables or disables automatic {@link #Include() Include}s.
<P>
	<B>N.B.</B>: This method controls the automatic include state for
	subsequent Configure operations. The {@link #Include_Default
	Include_Default} state is applied to the construction of new
	Configuration objects.
<P>
	@param	enable	true if automatic file includes is enabled;
		false otherwise.
	@return	The previous automatic include state.
*/
public boolean Include
	(
	boolean	enable
	)
{
boolean
	state = Include_Includes;
Include_Includes = enable;
return state;
}

/**	Enables or disables {@link #Include() Include} file tracing.
<P>
	Include file tracing inserts a TOKEN parameter at the beginning
	of the included parameters with a name of the form:
<P><BLOCKQUOTE>
	<B>INCLUDE_START:</B><I>source</I>
</BLOCKQUOTE><P>
	where <I>source</I> is the URL or canonical pathname of the file
	that was included. At the end of the included parameters another
	TOKEN is inserted that has the same form, but with "START" replaced
	by "END". If the source can not be found and the {@link
	#CONTINUE_INCLUDE_ON_ERROR_FLAG} is specified as the include file
	prefix, then a parameter with a name of the same form, but with
	"START" replaced by "ERROR", is inserted, and its value is the error
	message.
<P>
	<B>Warning</B>: When the resulting parameters are {@link #Coalesce()
	Coalesce}d the included parameters may be moved to new locations
	outside the INCLUDE_START to INCLUDE_END sequence. Thus include file
	tracing can only be advisory.
<P>
	<B>N.B.</B>: This method controls include file tracing for
	subsequent Configure operations. The {@link #Include_Tracing_Default
	Include_Tracing_Default} state is applied to the construction of
	new Configuration objects.
<P>
	@param	enable	true if include file tracing is enabled;
		false otherwise.
	@return	The previous include file tracing state.
*/
public boolean Include_Tracing
	(
	boolean	enable
	)
{
boolean
	state = Include_Tracing;
Include_Tracing = enable;
return state;
}

/**	Includes parameters from include files.
<P>
	Every parameter named {@link #INCLUDE} is replaced with the
	parameters included from the file named by the parameter's value.
	The value may contain nested parameter references that are {@link
	Reference_Resolver#Resolve(String) Resolve}d against the parameters
	currently in the Configuration; this includes all parameters that
	have been included so far.
<P>
	Relative pathnames for include files are taken to be relative to the
	directory of the file that contains the include parameter, if this
	can be determined. This will be the location of the Configuration
	{@link #Source() file} itself for top level include parameters.
<P>
	Include parameters may refer to a URL. If this is a "file" URL, then
	its directory can be used for relative includes contained within it.
	Other types of URLs may also be used, but they may not contain
	include parameters with relative filename references. Relative
	filename references that do not have a including file from which a
	directory can be determined (URLs or a Configuration that was not
	constructed from a file with an identified pathname) will be located
	using the same process as that used to {@link #Configure(String)
	Configure} from a source name.
<P>
	Recursive includes - an include parameter that refers to a parent
	include file - are not allowed (throw a Configuration_Exception).
<P>
	Parameters that are Aggregates or the Value is an Array are
	ignored.
<P>
	An include reference that begins with the {@link
	#CONTINUE_INCLUDE_ON_ERROR_FLAG} will not cause an exception to be
	thrown if the reference can not be accessed. This is particularly
	useful if the existence of the include file is uncertain and its
	lack is acceptable.
<P>
	@return	This Configuration.
	@throws	Configuration_Exception	If an include filename contains a
		nested reference that can not be resolved; would result in a
		recursive include; does not refer to a source that can be
		found, read or successfully parsed for parameters; or there was
		a problem coalescing its parameters (probably due to a
		duplicate parameter name).
*/
public synchronized Configuration Include ()
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_INCLUDE) != 0)
	System.out.println (">>> Configuration.Include");
String
	config_filename = null,
	filename,
	resolved_filename = null,
	included_pathname,
	name;

/*	Stack of included files and their parameters.
	The pathnames will be canonical pathnames or URLs.
*/
Stack
	included_pathnames = new Stack (),
	included_parameters = new Stack ();

//	Top level include file is the Configuration file itself (if known).
if ((filename = Source ()) != null &&
	! filename.equals (DEFAULTS))
	{
	if ((config_filename = the_filename (filename)) == null)
		config_filename = filename;	// A URL.
	if ((DEBUG & DEBUG_INCLUDE) != 0)
		System.out.println
			("    Top level (Configuration) include: " + config_filename);
	included_pathnames.push (config_filename);
	}
Reference_Resolver
	resolver = new Reference_Resolver ().Parameters (this);
Configuration
	include_parameters = null;
int
	include_index;
Parameter
	parent = null,
	parameter = null;
while ((parameter = Find (INCLUDE, false, parameter)) != null)
	{
	//	The Aggregate containing the include parameter.
	parent = parameter.Parent ();
	//	The index of the include parameter in it's parent's list.
	include_index = parent.getIndex (parameter);
	if ((DEBUG & DEBUG_INCLUDE) != 0)
		System.out.println
			("    Include found: " + parameter.Path_Name ());
	if (! included_parameters.empty () &&
		parameter == included_parameters.peek ())
		{
		/*	The include parameter is the last one found.

			This completes the search of the included parameters.
		*/
		if ((DEBUG & DEBUG_INCLUDE) != 0)
			System.out.println ("    Previous include.");
		//	Remove the include parameter.
		parent.Remove (include_index);
		//	Pop the included pathnames and parameters lists.
		included_pathnames.pop ();
		included_parameters.pop ();
		if (config_filename != null &&
			included_pathnames.empty ())
			//	Retain the Configuration pathname.
			included_pathnames.add (config_filename);
		/*
			Restart the search with the first parameter of
			the last included parameter's Aggregate list.
		*/
		parameter = (Parameter)parent.getChildAt (0);
		if ((DEBUG & DEBUG_INCLUDE) != 0)
			System.out.println
				("    Restarting search after " + parameter.Path_Name ());
		continue;
		}

	try
		{
		if (parameter.Is_Aggregate () ||
			parameter.Value ().Is_Array ())
			//	Ignore Aggregates and Arrays.
			continue;
		}
	catch (PVL_Exception exception) {/* Shouldn't happen */}

	//	Resolve any references in the include parameter value.
	try {resolved_filename = resolver.Resolve
			(filename = parameter.Value ().String_Data ());}
	catch (Throwable exception)
		{
		String
			message = ID + NL
				+ "Include of \"" + filename + '"'
				+ included_list (included_pathnames) + NL
				+ "contains a reference that could not be resolved." + NL
				+ exception.getMessage ();
		if (exception instanceof ParseException)
			message += NL
				+ "At location " +
					((ParseException)exception).getErrorOffset ();
		throw new Configuration_Exception (message, exception);
		}
	if ((DEBUG & DEBUG_INCLUDE) != 0)
		System.out.println
			("    Included: " + filename + NL
			+"    resolved: " + resolved_filename);

	//	Check for a continue-on-error flag.
	boolean
		continue_on_error = false;
	if (resolved_filename.startsWith (CONTINUE_INCLUDE_ON_ERROR_FLAG))
		{
		continue_on_error = true;
		resolved_filename = resolved_filename.substring
			(CONTINUE_INCLUDE_ON_ERROR_FLAG.length ());
		if ((DEBUG & DEBUG_INCLUDE) != 0)
			System.out.println
				("    continue: " + resolved_filename);
		}

	/*	Determine the include pathname.

		This will produce a canonical pathname for an absolute pathname
		to an existing file. Otherwise the name remains unchanged.

		When the name produces a non-file URL, null is returned to flag
		that no file checking can be done.
	*/
	included_pathname = the_filename (resolved_filename);
	if (included_pathname == null)
		{
		//	URL.
		if ((DEBUG & DEBUG_INCLUDE) != 0)
			System.out.println ("    URL");
		included_pathname = resolved_filename;
		}
	else
		{
		if ((DEBUG & DEBUG_INCLUDE) != 0)
			System.out.println ("    File");
		File
			file = new File (included_pathname);
		if (! included_pathnames.empty () &&
			! file.isAbsolute ())
			{
			name = the_filename
				((String)included_pathnames.peek ());
			if (name != null)
				//	Make the included filename relative to the includer file.
				included_pathname =
					new File (name).getParent () +
					File.separator +
					included_pathname;
			}
		if ((DEBUG & DEBUG_INCLUDE) != 0)
			System.out.println ("    Source: " + included_pathname);
		}
	if (Include_Tracing)
		{
		try
			{
			//	Insert INCLUDE_START before the include parameter.
			parent.Insert (new Parameter
				(INCLUDE_START + ':' + included_pathname), include_index++);
			//	Insert INCLUDE_END after the include parameter.
			parent.Insert (new Parameter
				(INCLUDE_END + ':' + included_pathname), include_index + 1);
			}
		catch (PVL_Exception exception) {/* Shouldn't happen */}
		}

	//	Load the parameters (turn off auto-include, of course, and Defaults).
	boolean
		auto_include = Include (false),
		defaults = Defaults_Default;
	Defaults_Default = false;
	try {include_parameters = new Configuration (included_pathname);}
	catch (Exception exception)
		{
		if (continue_on_error &&
			exception instanceof IllegalArgumentException)
			{
			//	Remove the INCLUDE parameter.
			if ((DEBUG & DEBUG_INCLUDE) != 0)
				System.out.println
					("    errored -" + NL
					+ exception);
			parent.Remove (include_index);
			if (Include_Tracing)
				{
				//	Insert an INCLUDE_ERROR parameter.
				try {parent.Insert (new Parameter
						(INCLUDE_ERROR + ':' + included_pathname)
						.Value (exception.getMessage ()), include_index);}
				catch (PVL_Exception except) {/* Shouldn't happen */}
				}
			//	Restart the search with the parent's first parameter.
			parameter = (Parameter)parent.getChildAt (0);
			continue;
			}
		throw new Configuration_Exception
			(
			ID + NL
			+ "Include of \"" + included_pathname + '"'
			+ (resolved_filename.equals (filename) ?
				"" : (NL + "resolved from \"" + filename + '"'))
			+ included_list (included_pathnames) + NL
			+ "is unable to obtain the parameters."
			+ exception_message (exception),
			exception
			);
		}
	finally
		{
		Include (auto_include);
		Defaults_Default = defaults;
		}

	//	Check for recursive file include.
	if ((included_pathname = the_filename (include_parameters.Source ()))
			== null)
		included_pathname = include_parameters.Source ();	//	URL.
	if (included_pathnames.contains (included_pathname))
		throw new Configuration_Exception
			(
			ID + NL
			+ "Include of \"" + included_pathname + '"'
			+ (resolved_filename.equals (filename) ?
				"" : (NL + "resolved from \"" + filename + '"'))
			+ included_list (included_pathnames) + NL
			+ "would be recursive."
			);

	if (include_parameters.getChildCount () == 0)
		{
		//	Empty include parameters list.
		if ((DEBUG & DEBUG_INCLUDE) != 0)
			System.out.println ("    Include list is empty.");
		parent.Remove (include_index);
		parameter = null;
		continue;
		}

	//	Insert the new parameters before the include parameter.
	try {parent.Insert
			(include_parameters.List (), include_index);}
	catch (PVL_Exception exception) 
		{
		throw new Configuration_Exception
			(
			ID + NL
			+ "Include of \"" + included_pathname + '"'
			+ (resolved_filename.equals (filename) ?
				"" : (NL + "resolved from \"" + filename + '"'))
			+ included_list (included_pathnames) + NL
			+ "was unable to insert the new parameters!"
			+ exception_message (exception),
			exception
			);
		}
	/*
		Note that the include parameter is left in place to mark the end
		of the included parameters during the search for more include
		parameters in the included parameters.
	*/
	if ((DEBUG & DEBUG_INCLUDE) != 0)
		System.out.println
			("    Included parameters:" + NL
			+ include_parameters.Description ());

	//	Add the included pathname to the list.
	included_pathnames.push (included_pathname);
	included_parameters.push (parameter);

	//	Restart the search with the first Parameter in the current Aggregate.
	parameter = (Parameter)parent.getChildAt (0);
	}
if ((DEBUG & DEBUG_INCLUDE) != 0)
	System.out.println ("<<< Configuration.Include");
return this;
}

/**	Generates a listing of included files.
<P>
	The listing is LIFO: the last file included is listed first
	followed, on succeeding lines, by the file that included the
	previous file. Each filename is enclosed in quotes and preceded by a
	newline and "included from ". If the list of included files is
	empty, the empty String is returned.
<P>
	<b>N.B.</b>: The listing does not end with a new-line character.
<P>
	@param	included_pathnames	The Vector of included filenames.
	@return	The included filenames listing String.
*/
private static String included_list
	(
	Stack	included_pathnames
	)
{
if (included_pathnames.isEmpty ())
	return "";
String
	included = "";
ListIterator
	names = included_pathnames.listIterator (included_pathnames.size ());
while (names.hasPrevious ())
	included += NL
		+ "included from \"" + (String)(names.previous ()) + '"';
return included;
}

/**	Gets a filename from what may be a URL.
<P>
	An attempt is made to construct a URL from the argument. If this
	succeeds the protocol is checked to see if it is a "file". If so
	the canononical path from the URL is returned, otherwise null is
	returned. If the URL can not be constructed or the canonical path
	to an existing file can not be determined from a file URL the
	original argument is returned.
<P>
	@param	string	The initial filename.
	@return	A String, or null if the string is a non-file URL.
*/
private static String the_filename
	(
	String	string
	)
{
try
	{
	URL
		url = new URL (string);
	if (url.getProtocol ().equals ("file"))
		{
		File
			file = new File (string = url.getPath ());
		if (file.exists ())
			string = file.getCanonicalPath ();
		// else it's not an existing file.
		}
	else
		//	It's a non-file URL.
		string = null;
	}
catch (MalformedURLException exception)
	{
	//	Not a URL.
	File
		file = new File (string);
	if (file.isAbsolute () &&
		file.exists ())
		{
		try {string = file.getCanonicalPath ();}
		catch (IOException except) {}
		}
	}
catch (IOException exception) {/* Can't get the cononical path */}
return string;
}

/*==============================================================================
	Utility functions
*/
/**	Gets the pathname for a parameter at the previous group level.
<P>
	The {@link #Basename(String) Basename} is removed from the pathname.
	Then the pathname is shortened by one additional segment and the
	basename appended back again. This has the effect of producing a
	pathname with the same basename but at the same level as the parent
	(thus it is the "uncle" to the original pathname :-). This is used
	when percolating up a Parameter hierarchy looking for an effective
	(default) parameter of a given name.
<P>
	@param	pathname	The pathname to modify.
	@return	The pathname with the parent segment removed, or null
		if the pathname has no parent segment.
*/
public static String Pathname_Uncle
	(
	String	pathname
	)
{
if (pathname == null)
	return null;
int
	end = pathname.lastIndexOf (Path_Delimiter ());
if (end <= 0)
	return null;
int
	start = pathname.lastIndexOf (Path_Delimiter (), end - 1);
if (start < 0)
	return pathname.substring (++end);
if (start == 0)
	return pathname.substring (end);
return pathname.substring (0, start) + pathname.substring (end);
}

/**	Gets the Configuration pathname for a Parameter Path_Name.
<P>
	A Parameter {@link Parameter#Path_Name() Path_Name} includes the
	name of the root Aggregate parameter. A Configuration pathname does
	not include the root name segment.
<P>
	@param	pathname	A Parameter Path_Name.
	@return	The pathname stripped of the root segment, or the empty
		String if the pathname only has a root segment.
*/
public static String Path_Name_to_Pathname
	(
	String	pathname
	)
{
if (pathname == null)
	return null;
int
	end = pathname.indexOf (Path_Delimiter (), 1);
if (end > 0)
	return pathname.substring (end);
return "";
}

/**	Gets the Configuration pathaname for a parameter.
<P>
	@param	parameter	A parameter.
	@return	The pathname String for the parameter.
	@see	#Path_Name_to_Pathname(String)
*/
public static String Pathname
	(
	Parameter	parameter
	)
{return Path_Name_to_Pathname (parameter.Path_Name ());}

/**	Gets the effective pathname for a specified pathname.
<P>
	The effective pathname is the pathname of the {@link
	#Effective_Parameter(String) Effective_Parameter} for the pathname.
<P>
	@param	pathname	A parameter pathname.
	@return	The effective pathname for the pathname, or null if there
		is none.
*/
public String Effective_Pathname
	(
	String	pathname
	)
{
Parameter
	parameter = Effective_Parameter (pathname);
if (parameter != null)
	return parameter.Path_from (this);
return null;
}

/**	Gets the pathname to a parameter group.
<P>
	Links will be followed: When an Assignment Parameter with the name
	of the group is found its value will be taken as the new group name
	to be sought until an Aggregate Parameter is found. A link chain is
	not allowed to loop back to the original group name.
<P>
	@param	group	The name of an Aggregate Parameter to be found.
	@return	The fully qualified pathname to the parameter group,
		or null if the group can not be found.
*/
public String Group_Pathname
	(
	String	group
	)
{
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (">>> Configuration.Group_Pathname: " + group);
if (group == null)
	{
	if ((DEBUG & DEBUG_UTILITY) != 0)
		System.out.println ("<<< Configuration.Group_Pathname: " + group);
	return null;
	}
Parameter
	parameter = null,
	pattern = new Parameter (group);
Selector
	selection = new Selection ()
		.Name (true)
		.Specific (Case_Sensitive);
String
	link = group;
while ((parameter = Find (pattern, selection, parameter)) != null &&
	  ! parameter.Is_Aggregate ())
	{
	if (parameter.Is_Assignment ())
		{
		try
			{
			//	Follow links.
			if (parameter.Value () != null &&
				! (link = parameter.Value ().String_Data ()).equals ("") &&
				! (Case_Sensitive ?
					link.equals (group) : link.equalsIgnoreCase (group)))
				{
				//	Find the new name from the top.
				pattern.Name (link);
				parameter = null;
				if ((DEBUG & DEBUG_UTILITY) != 0)
					System.out.println ("    link: " + link);
				}
			}
		catch (PVL_Exception exception) {/* Shouldn't happen */ return null;}
		}
	}
if (parameter != null)
	link = Path_Name_to_Pathname (parameter.Path_Name ());
else
	link = null;
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println ("<<< Configuration.Group_Pathname: " + link);
return link;
}

/**	Gets the parameter for a pathname.
<P>
	When specific is false and there is no parameter in the
	Configuration at the specified pathname location, then a parameter
	with the same {@link #Basename(String) Basename} is sought at
	progressively higher levels of the path. The first parameter found
	is returned, or null if no parameter is found.
<P>
	<B>Note</B>: Only an Assignment (or Token) parameter will be found.
	Aggregate parameters do not qualify because they have no value. A
	Token has no value but can be assigned one.
<P>
	@param	pathname	A parameter pathname.
	@param	specific	true if only a parameter at the specific
		pathname is acceptable, otherwise an effective parameter will
		do. 
	@return	The Parameter for the pathname, or null if there is none.
	@see	#Pathname_Uncle(String)
*/
public Parameter Parameter
	(
	String	pathname,
	boolean	specific
	)
{return Parameter (pathname, specific, 0);}


/**	Gets the Nth parameter for a pathname.
<P>
	The parameter is selected after skip parameters have been found
	that match the pathname. If skip is 0, the first matching parameter
	is selected; if skip is N, N matching parameters must be found
	before the next matching parameter will be selected.
<P>
	@param	pathname	A parameter pathname.
	@param	specific	true if only a parameter at the specific
		pathname is acceptable, otherwise an effective parameter will
		do.
	@param	skip		The number of matching parameters to skip.
	@return	The Parameter for the pathname, or null if there is none
		found.
	@see	#Parameter(String, boolean)
*/
public Parameter Parameter
	(
	String	pathname,
	boolean	specific,
	int		skip
	)
{
if (pathname == null)
	return null;
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (">>> Configuration.Parameter (\"" +
		pathname + "\", " + specific + ")");
//	Find only non-Aggregate parameters with the specified name.
Parameter
	pattern = new Parameter (pathname);
Selector
	selection = new Selection ()
		.Name (true)
		.Specific (Case_Sensitive);
Parameter
	parameter = null;
do
	{
	while ((parameter = Find (pattern, selection, parameter)) == null &&
			! specific)
		{
		if ((pathname = Pathname_Uncle (pathname)) == null)
			break;
		if ((DEBUG & DEBUG_UTILITY) != 0)
			System.out.println ("    Trying Pathname_Uncle: " + pathname);
		pattern.Name (pathname);
		}
	}
	//	Accept only the Nth non-Aggregate parameter.
	while (parameter != null &&
		(parameter.Is_Aggregate () || skip-- > 0));
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println ("<<< Configuration.Parameter: " + 
		(parameter == null ? "not found." : "found"));
return parameter;
}

/**	Gets the parameter or parameter group at a pathname.
<P>
	Only the parameter at the specific pathname will be found. This is
	similar to the {@link #Parameter(String, boolean) Parameter} method
	used with specific=true. But unlike that method, this method may
	return an Aggregate if that is what is found.
<P>
	@param	pathname	A parameter pathname.
	@return	The Parameter for the pathname, or null if there is none.
*/
public Parameter Parameter_at
	(
	String	pathname
	)
{
if (pathname == null)
	return null;
Parameter
	pattern = new Parameter (pathname);
Selector
	selection = new Selection ()
		.Name (true)
		.Specific (Case_Sensitive);
return Find (pattern, selection);
}

/**	Gets the specific Parameter for a pathname.
<P>
	This is the same as using the {@link #Parameter(String,
	boolean) Parameter} method with the specfic
	argument being true.
<P>
	@param	pathname	A parameter pathname.
	@return	The Parameter for the pathname, or null if there is none.
*/
public Parameter Specific_Parameter
	(
	String	pathname
	)
{return Parameter (pathname, true);}

/**	Gets the effective Parameter for a pathname.
<P>
	This is the same as using the {@link #Parameter(String, boolean)
	Parameter} method with the specific argument being false.
<P>
	@param	pathname	A parameter pathname.
	@return	The effective Parameter for the pathname, or null if there
		is none.
*/
public Parameter Effective_Parameter
	(
	String	pathname
	)
{return Parameter (pathname, false);}

/**	Sets the duplicate parameter action during coalescing.
<P>
	One of these actions may be specified:
<P>
<DL>
	<DT>PREFER_FIRST_PARAMETER (>0)
		<DD>When a duplicate Assignment Parameter pathname is found
		while coalescing the Configuration, give preference to the
		first parameter; i.e. remove all but the first of any duplicate
		pathname parameters.
	<DT>PREFER_LAST_PARAMETER (<0)
		<DD>When a duplicate Assignment Parameter pathname is found
		while coalescing the Configuration, give preference to the last
		parameter; i.e. remove all but the last of any duplicate
		pathname parameters.
	<DT>THROW_ON_DUPLICATE_PARAMETERS (0)
		<DD>When a duplicate Assignment Parameter pathname is found
		while coalescing the Configuration, a Configuration_Exception
		will be thrown.
</DL>
<P>
	The default action is to THROW_ON_DUPLICATE_PARAMETERS.
<P>
	@param	action	The duplicate parameter action.
	@return	This Configuration
	@see	#Coalesce()
*/
public Configuration Duplicate_Parameter_Action
	(
	int		action
	)
{
if (action == 0)
	Duplicate_Parameter_Action = THROW_ON_DUPLICATE_PARAMETERS;
else if (action > 0)
	Duplicate_Parameter_Action = PREFER_FIRST_PARAMETER;
else if (action < 0)
	Duplicate_Parameter_Action = PREFER_LAST_PARAMETER;
return this;
}

/**	Gets the duplicate parameter action during coalescing.
<P>
	@return	The default parameter action.
	@see	#Duplicate_Parameter_Action(int)
*/
public int Duplicate_Parameter_Action ()
{return Duplicate_Parameter_Action;}

/**	Coalesce all Aggregate parameters of the same name together.
<P>
	Each Aggregate in the current list is compared, by name, against
	all other Aggregates in the list. When a duplicate name is found
	the parameters it contains are added to the end of the first
	parameter in the list having the same name. Finally each Aggregate
	is recursively coalesced in the same manner.
<P>
	In addition, all Assignment (i.e. non-Aggregate) parameters in each
	Aggregate are checked for duplicates and moved to the front of the
	list of parameters.
<P>
	The action taken when duplicate Assignment Parameter is encountered
	depends on the current {@link #Duplicate_Parameter_Action(int)
	Duplicate_Parameter_Action}. Byt default the actions is to throw
	a Configuration_Exception.
<P>
	@throws	Configuration_Exception	If there was a duplicate Assignment
		parameter and the Duplicate_Parameter_Action is
		THROW_ON_DUPLICATE_PARAMETERS, or there was a problem moving
		parameters around in the list.
*/
public void Coalesce ()
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (">>> Coalesce");
coalesce (this);
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println ("<<< Coalesce");
}

private synchronized void coalesce
	(
	Parameter	parameters
	)
	throws Configuration_Exception
{
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println (">>> coalesce: " + parameters.Path_Name ());
Vector
	list;
try {
	list = parameters.List ();
	if (list == null)
		{
		if ((DEBUG & DEBUG_UTILITY) != 0)
			System.out.println ("<<< coalesce: " + parameters.Path_Name ());
		return;
		}
	}
catch (PVL_Exception exception)
	{
	/* No list */
	if ((DEBUG & DEBUG_UTILITY) != 0)
		System.out.println ("<<< coalesce: " + parameters.Path_Name ());
	return;
	}
int
	this_index,
	that_index,
	total_parameters = list.size ();
if (total_parameters == 0)
	{
	if ((DEBUG & DEBUG_UTILITY) != 0)
		System.out.println ("<<< coalesce: " + parameters.Path_Name ());
	return;
	}
Parameter
	this_parameter,
	that_parameter,
	/*	Accumulation of Assignment Parameters in the current Parameter list.
		It starts as an Assignment as a convenient flag that it is empty.
	*/
	assignments = new Parameter ();

for (this_index = 0;
	 this_index < total_parameters;
	 this_index++)
	{
	this_parameter = (Parameter)list.elementAt (this_index);
	if (this_parameter.Is_Aggregate ())
		{
		for (that_index = this_index + 1;
			 that_index < total_parameters;
			 that_index++)
			{
			//	Check the remainder of the list for matching Aggregates.
			that_parameter = (Parameter)list.elementAt (that_index);
			if (that_parameter.Is_Aggregate () &&
				(Case_Sensitive ?
					that_parameter.Name ()
						.equals (this_parameter.Name ()) :
					that_parameter.Name ()
						.equalsIgnoreCase (this_parameter.Name ())))
				{
				try
					{
					this_parameter.Add
						(parameters.Remove (that_index--).List ());
					total_parameters--;
					}
				catch (PVL_Exception exception)
					{
					throw new Configuration_Exception
						(
						ID + NL
						+ "During Coalesce of \"" + parameters.Name () + '"' + NL
						+ "  unable to Add to \""
							+ this_parameter.Name ()
							+ "\" the parameter List from \""
							+ that_parameter.Name () + "\"."
						+ exception_message (exception),
						exception
						);
					}
				}
			}
		//	Coalesce the coalesced Aggregate.
		coalesce (this_parameter);
		}
	else
		{
		//	Check for a duplicate Assignment parameter.
		if (assignments.Is_Aggregate () &&
			(that_parameter = assignments.Find
					(this_parameter.Name (), Case_Sensitive))
				!= null)
			{
			switch (Duplicate_Parameter_Action)
				{
				case PREFER_FIRST_PARAMETER:
					//	Drop the duplicate from the parameters list.
					parameters.Remove (this_index--);
					total_parameters--;
					//	Move on to the next parameter in the list.
					continue;
				case THROW_ON_DUPLICATE_PARAMETERS:
					String
						this_value = null,
						that_value = null;
					try
						{
						this_value = this_parameter.Value ().toString ();
						that_value = that_parameter.Value ().toString ();
						}
					catch (PVL_Exception exception) {/* Shouldn't happen. */}
					throw new Configuration_Exception
						(
						ID + NL
						+ "During Coalesce of \"" + parameters.Name () + '"' + NL
						+ "  duplicate parameter name - \""
							+ this_parameter.Path_Name () + "\";" + NL
						+ "   original value - " + that_value + NL
						+ "  duplicate value - " + this_value
						);
				case PREFER_LAST_PARAMETER:
					//	Drop the assignments list duplicate.
					assignments.Remove (that_parameter);
					//	Accumulate the parameters list duplicate.
				}
			}
		try
			{
			//	Move the the Assignment Parameter into the accumulated list.
			assignments.Add
				(parameters.Remove (this_index--));
			total_parameters--;
			}
		catch (PVL_Exception exception)
			{
			throw new Configuration_Exception
				(
				ID + NL
				+ "During Coalesce of \"" + parameters.Name () + '"' + NL
				+ "  while accumulating Assignment parameter \""
					+ this_parameter.Name () + "\"." + NL
				+ exception_message (exception),
				exception
				);
			}
		}
	}
if (assignments.Is_Aggregate ())
	{
	//	Insert the accumulated Assignments before the first Aggregate.
	try {parameters.Insert (assignments.List (), 0);}
	catch (PVL_Exception exception)
		{
		throw new Configuration_Exception
			(
			ID + NL
			+ "During Coalesce of \"" + parameters.Name () + '"' + NL
			+ "  unable to move Assignment parameters to the front."
			+ exception_message (exception),
			exception
			);
		}
	}
if ((DEBUG & DEBUG_UTILITY) != 0)
	System.out.println ("<<< coalesce: " + parameters.Path_Name ());
}

/*==============================================================================
	Helpers
*/
/**	Get an exception message.

	@param	exception	The Exception from which to obtain a message.
	@return	A String that begins with a NL sequence followed by the
		message from the exception if the exception is not null and its
		message is neither null nor empty.
*/
static private String exception_message
	(
	Throwable	exception
	)
{
String
	message = NL;
if (exception != null)
	{
	String
		exception_message = exception.getMessage ();
	if (exception_message != null &&
		exception_message.length () != 0)
		message += exception_message;
	}
return message;
}

/*==============================================================================
	Application
*/
/**	Assemble and report on the contents of a configuration.
<P>
	The command line syntax is described by the {@link #Usage()
	Usage} method.
<P>
	Exit status:
<P>
	0 - Success<br>
	1 - Command line syntax problem<br>
	2 - Unable to assemble the configuration<br>
	3 - The source file could not be found<br>
*/
public static void main
	(
	String[] arguments
	)
{
if (arguments.length == 0)
	Usage ();

Configuration
	configuration = null;
String
	source = null;
Vector
	find_pathnames = new Vector ();
boolean
	quiet          = false,
	case_sensitive = false,
	pattern_match  = false;

for (int count = 0;
		 count < arguments.length;
		 count++)
	{
	if (arguments[count].length () > 1 &&
		arguments[count].charAt (0) == '-')
		{
		switch (arguments[count].charAt (1))
			{
			case 'Q':	//	Quiet
			case 'q':
				quiet = true;

			case 'F':	//	Find
			case 'f':
				if (++count == arguments.length ||
					arguments[count].charAt (0) == '-')
					{
					System.out.println
						("Missing parameter pathname.");
					Usage ();
					}
				find_pathnames.add (arguments[count]);
				break;

			case 'C':	//	Case_sensitive
			case 'c':
				case_sensitive = true;
				pattern_match  = false;
				break;

			case 'N':	//	Not_case_sensitive
			case 'n':
				case_sensitive = false;
				break;

			case 'P':	//	Pattern
			case 'p':
				pattern_match  = true;
				case_sensitive = false;
				break;

			case 'V':	//	Version
			case 'v':
				System.out.println (ID);
				System.exit (0);

			default:
				System.out.println
					("Unknown option: " + arguments[count]);

			case 'H':	//	Help
			case 'h':
				Usage ();
			}
		}
	else
		{
		if (source != null)
			{
			System.out.println
				("More than one Configuration source specified -" + NL
				+"   First source: " + source + NL
				+"  Second source: " + arguments[count]);
			Usage ();
			}
		source = arguments[count];
		}
	}
if (source == null)
	{
	System.out.println ("No Configuration source specified.");
	Usage ();
	}

try
	{
	Include_Tracing_Default = true;
	try
		{
		if (source.equals ("-"))
			configuration = new Configuration (System.in);
		else
			configuration = new Configuration (source);
		}
	catch (IllegalArgumentException exception)
		{
		System.err.println
			("Could not find the configuration source file: " + source);
		System.exit (3);
		}
	configuration.Add_Environment ();

	if (find_pathnames.isEmpty ())
		System.out.println (configuration.Description ());
	else
		{
		Iterator
			pathnames = find_pathnames.iterator ();
		String
			parent_pathname,
			comments;
		Parameter
			parameter,
			test_parameter = new Parameter ();
		Selection
			selection = new Selection ();
		selection.Name (true);
		selection.Specific (case_sensitive);
		selection.Pattern_Match (pattern_match);
		Lister
			lister = new Lister ().Indent_Arrays (false);
		while (pathnames.hasNext ())
			{
			test_parameter.Name ((String)pathnames.next ());
			parameter = null;
			while ((parameter = configuration.Find
					(test_parameter, selection, parameter)) != null)
				{
				comments = parameter.Comments ();
				parameter.Comments (null);
				try
					{
					if (! quiet &&
						comments != null)
						lister.Write_Comments (comments);
					parent_pathname = Parent_Pathname (Pathname (parameter));
					if (parent_pathname != null)
						System.out.print (parent_pathname);
					System.out.print (Path_Delimiter ());
					parameter.Write (lister);
					}
				catch (PVL_Exception exception)
					{
					System.err.println
						("PVL_Exception while listing parameter \""
							+ parameter.Path_Name () + '"' + NL
						+ exception.getMessage ());
					}
				catch (IOException exception)
					{
					System.err.println
						("IOException while listing parameter \""
							+ parameter.Path_Name () + '"' + NL
						+ exception.getMessage ());
					}
				}
			}
		}
	}
catch (Configuration_Exception exception)
	{
	System.err.println
		(NL
		+"Configuration Exception:" + NL
		+ exception.getMessage ());
	System.exit (2);
	}
System.exit (0);
}


/**	Prints the command line usage syntax.
<P>
<BLOCKQUOTE><PRE>
Usage: <B>Configuration</B> &lt;<I>Options</I>&gt; &lt;<I>source</I>&gt;
&nbsp;&nbsp;Options -
&nbsp;&nbsp;&nbsp;&nbsp;<B>-<U>Q</U>uery</B> | <B>-<U>F</U>ind</B> &lt;<I>pathname</I>&gt;
&nbsp;&nbsp;&nbsp;&nbsp;[<B>-<U>N</U>ot_</B>]<B><U>C</U>ase_sensitive</B>
&nbsp;&nbsp;&nbsp;&nbsp;<B>-<U>P</U>attern</B>
</PRE></BLOCKQUOTE>
<P>
	Only the source of the configuration parameters is required. This
	may be a filename or a URL.
<P>
	The entire configuration, with all included configurations, will be
	listed unless a specific parameter pathname is specified with the
	-query or -find option. The only difference between -query and -find
	is that the former never lists parameter comments. A parameter
	pathname may be absolute or relative. All parameters having the
	pathname will be listed. Pathname matching is not case sensitive
	unless the -case_sensitive option is specified. If -pattern is
	specified the pathname is taken to be a regular expression
	{@link java.util.regex.Pattern Pattern}. The -query/-find option may be
	used more than once to find more than one parameter. 
<P>
	<b>N.B.</b>This method always results in a System.exit with a
	status of 1.
*/
public static void Usage ()
{
System.out.print
	(ID + NL
	+"Usage: Configuration <Options> <source>" + NL
	+"  Options -" + NL
	+"    -Query | -Find <pathname>" + NL
	+"    -[Not_]Case_sensitive" + NL
	+"    -Pattern" + NL
	+"    -Help" + NL
	+"    -Version" + NL
	);
System.exit (1);
}


}	//	End of class
