//
// =================================================================
//
//  MacroMolecular Data Base (MMDB)
//
//  File  extr_exmp1.cpp
//
//  Example:
//     Extraction of atoms from MMDB.
//
//  16 Feb 2001
//
//  SGI make:  f77 -o extr_exmp1 extr_exmp1.cpp mmdb.a -lm -lC
//
// =================================================================
//

#include <cstring>
#include "mmdb_manager.h"


void  PrintInstructions()  {
  printf ( 
    "\n"
    "A MacroMolecular Data Base (MMDB) example:\n"
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "Extraction of atoms from MMDB.\n"
    "\n"
    "Command line syntax:\n"
    "~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "extr_exmp1 coor_file\n"
    "\n"
    "where 'coor_file' is a coordinate file in PDB, mmCIF or\n"
    "BINary format.\n"
    "\n"
         );
}


int main ( int argc, char ** argv, char ** env )  {
PCMMDBManager  MMDBManager;
int            RC,lcount;
char           S[500];
PCAtom         A1,A2;

  if (argc<2)  {
    PrintInstructions();
    return 1;
  }

  //  1.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();

  MMDBManager = NULL;  // all pointers should be either NULL or allocated

  //  2.  Create an instance of MMDB
  MMDBManager = new CMMDBManager();

  //  3.  Read coordinate file.
  //    3.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed
  MMDBManager->SetFlag ( MMDBF_PrintCIFWarnings );

  //    3.2 MMDB provides the following function for reading
  //        coordinate files with automatical format recognition:
  RC = MMDBManager->ReadCoorFile ( argv[1] );

  //    3.3 Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDBManager->GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  dispose instance of MMDB and quit:
    delete MMDBManager;
    return 2;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (MMDBManager->GetFileType())  {
      case MMDB_FILE_PDB    :  printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    :  printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary :  printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",argv[1] );
  }


  //  4.  Extract atoms. Each atom is identified by model number
  //      (1 if no models mentioned in coordinate file), chain ID
  //      within the model, residue number and insertion code
  //      within the chain and atom name (and chemical element name)
  //      and alternate location indicator within the residue.

  //    4.1 GetAtom() has two forms. The first one takes the
  //        atom identifying data from individual parameters
  //        (check in file mmdb_coormngr.h).
  A1 = MMDBManager->GetAtom ( 1,"A",33,"","CA",NULL,"" );
  if (!A1)  {
    // atom not found, identify the reason:
    switch (MMDBManager->CoorIDCode)  {
      case CID_NoModel   : strcpy ( S,"Model not found" );            break;
      case CID_NoChain   : strcpy ( S,"Chain not found in model" );   break;
      case CID_NoResidue : strcpy ( S,"Residue not found in chain" ); break;
      case CID_NoAtom    : strcpy ( S,"Atom not found in residue" );  break;
      default            : strcpy ( S,"Unknown error code" );
    }
    printf ( " failed on atom #1: %s\n",S );
  } else
    // class CAtom (cf. file mmdb_atom.h) allows for direct access
    // to atom parameters. It is Ok to read them, but editing
    // operations should be done through the mmdb interface to
    // ensure that internal links and tables are maintained
    // properly.
    printf ( " atom #1: x=%12g  y=%12g  z=%12g\n",
             A1->x,A1->y,A1->z );

  //    4.2 Second form of GetAtom() takes the atom identifying data
  //        from a string of a special syntax. Check this in file
  //        mmdb_coormngr.h.
  A2 = MMDBManager->GetAtom ( "/1/B/131/C5" );
  if (!A2)  {
    // atom not found, identify the reason:
    switch (MMDBManager->CoorIDCode)  {
      case CID_NoModel   : strcpy ( S,"Model not found" );            break;
      case CID_NoChain   : strcpy ( S,"Chain not found in model" );   break;
      case CID_NoResidue : strcpy ( S,"Residue not found in chain" ); break;
      case CID_NoAtom    : strcpy ( S,"Atom not found in residue" );  break;
      case CID_WrongPath : strcpy ( S,"Wrong syntax of atom path" );  break;
      default              : strcpy ( S,"Unknown error code" );
    }
    printf ( " failed on atom #2: %s\n",S );
  } else
    // see remark in similar place above.
    printf ( " atom #2: x=%12g  y=%12g  z=%12g\n",
             A2->x,A2->y,A2->z );



  //  5. A formally correct style of programming requires you
  //     to explicitely release any allocated memory before
  //     your application quits.
  if (MMDBManager)  delete MMDBManager;

  return 0;

}

