/*
 *  $Id: mfmops.h 20788 2018-02-02 09:41:17Z yeti-dn $
 *  Copyright (C) 2017 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef __GWY_PROCESS_MFMOPS_H__
#define __GWY_PROCESS_MFMOPS_H__

#include <libprocess/gwyprocesstypes.h>
#include <libprocess/arithmetic.h>
#include <libprocess/inttrans.h>
#include <libprocess/filters.h>
#include <libprocess/stats.h>

#define MU_0 1.256637061435917295e-6
#define EPSILON_0 8.854187817620389850e-12

#define MFM_DIMENSION_ARGS_INIT \
    { 256, 256, 5.0, (gpointer)"m", NULL, -9, 0, FALSE, FALSE }

typedef enum {
    GWY_MFM_PROBE_CHARGE = 0,
    GWY_MFM_PROBE_BAR    = 1
} GwyMfmProbeType;

typedef enum {
    GWY_MFM_COMPONENT_HX = 0,
    GWY_MFM_COMPONENT_HY = 1,
    GWY_MFM_COMPONENT_HZ = 2
} GwyMfmComponentType;

G_GNUC_UNUSED
static void
mfm_perpendicular_create_field_mask(GwyDataField *fieldmask, gdouble height,
                                    gdouble thickness)
{
    gint i, j;
    gdouble kx, ky, k;
    gdouble *data = gwy_data_field_get_data(fieldmask);

    gint xres = gwy_data_field_get_xres(fieldmask);
    gint yres = gwy_data_field_get_yres(fieldmask);
    gdouble xreal = gwy_data_field_get_xreal(fieldmask);
    gdouble yreal = gwy_data_field_get_yreal(fieldmask);

    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {

            kx = ABS(i - xres/2)/xreal;
            ky = ABS(j - yres/2)/yreal;

            k = hypot(kx, ky);

            data[i*xres + j] = 0.5*(exp(-k*height)*(1.0-exp(-k*thickness)));

        }
    }
}

G_GNUC_UNUSED
static void
mfm_perpendicular_create_wall_mask(GwyDataField *wm, gdouble a, gdouble kn)
{
    gint i, j;
    gdouble x, y, s, *data;
    gdouble delta = G_PI*sqrt(a/kn);

    gint xres = gwy_data_field_get_xres(wm);
    gint yres = gwy_data_field_get_yres(wm);
    gdouble xreal = gwy_data_field_get_xreal(wm);
    gdouble yreal = gwy_data_field_get_yreal(wm);

    data = gwy_data_field_get_data(wm);
    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {

            x = ABS(i - xres/2)*xreal/(gdouble)xres;
            y = ABS(j - yres/2)*yreal/(gdouble)yres;

            s = hypot(x, y);

            data[i*xres + j] = 1.0/(cosh(G_PI*s/delta));
        }
    }
}

G_GNUC_UNUSED
static inline gdouble
mysinc(gdouble x)
{
    if (x == 0.0)
        return 1;
    else
        return sin(x)/x;
}

G_GNUC_UNUSED
static void
mfm_perpendicular_create_ftf(GwyDataField *ftf, gdouble mtip,
                             gdouble bx, gdouble by, gdouble length,
                             GwyMfmProbeType type)
{
    gint i, j;
    gdouble kx, ky, k, *data;


    gint xres = gwy_data_field_get_xres(ftf);
    gint yres = gwy_data_field_get_yres(ftf);
    gdouble xreal = gwy_data_field_get_xreal(ftf);
    gdouble yreal = gwy_data_field_get_yreal(ftf);

    if (type == GWY_MFM_PROBE_CHARGE)
        gwy_data_field_fill(ftf, -MU_0*mtip*bx*by);
    else {
        data = gwy_data_field_get_data(ftf);
        for (i = 0; i < yres; i++) {
            for (j = 0; j < xres; j++) {
                kx = ABS(i - xres/2)/xreal;
                ky = ABS(j - yres/2)/yreal;

                k = hypot(kx, ky);

                data[i*xres + j] = -MU_0*mtip*bx*by*mysinc(kx*bx/2)*mysinc(ky*by/2)
                                   *(1-exp(-k*length));
            }
        }
    }
}

G_GNUC_UNUSED
static void
mfm_perpendicular_stray_field(GwyDataField *mfield, GwyDataField *out,
                              gdouble height, gdouble thickness,
                              gdouble sigma,
                              gboolean walls, gdouble wall_a, gdouble wall_kn)
{
    GwyDataField *rea, *ima, *reb, *imb, *fieldmask, *wallmask;

    rea = gwy_data_field_new_alike(mfield, TRUE);
    reb = gwy_data_field_new_alike(mfield, TRUE);
    ima = gwy_data_field_new_alike(mfield, TRUE);
    imb = gwy_data_field_new_alike(mfield, TRUE);
    fieldmask = gwy_data_field_new_alike(mfield, TRUE);
    wallmask = gwy_data_field_new_alike(mfield, TRUE);

    gwy_data_field_copy(mfield, rea, FALSE);
    gwy_data_field_multiply(rea, 2*sigma);
    gwy_data_field_add(rea, -sigma);

    if (walls) {
        mfm_perpendicular_create_wall_mask(wallmask, wall_a, wall_kn);
        gwy_data_field_area_ext_convolve(rea,
                                         0, 0,
                                         gwy_data_field_get_xres(rea),
                                         gwy_data_field_get_yres(rea),
                                         rea, wallmask,
                                         GWY_EXTERIOR_MIRROR_EXTEND, 0.0,
                                         FALSE);
    }

    gwy_data_field_2dfft_raw(rea, NULL, reb, imb,
                             GWY_TRANSFORM_DIRECTION_FORWARD);

    gwy_data_field_2dfft_humanize(reb);
    gwy_data_field_2dfft_humanize(imb);

    mfm_perpendicular_create_field_mask(fieldmask, height, thickness);

    gwy_data_field_multiply_fields(reb, reb, fieldmask);
    gwy_data_field_multiply_fields(imb, imb, fieldmask);

    gwy_data_field_2dfft_dehumanize(reb);
    gwy_data_field_2dfft_dehumanize(imb);
    gwy_data_field_2dfft_raw(reb, imb, rea, ima,
                             GWY_TRANSFORM_DIRECTION_BACKWARD);

    gwy_data_field_copy(rea, out, FALSE);

    gwy_object_unref(rea);
    gwy_object_unref(reb);
    gwy_object_unref(ima);
    gwy_object_unref(imb);
    gwy_object_unref(fieldmask);
    gwy_object_unref(wallmask);
}

G_GNUC_UNUSED
static void
mfm_perpendicular_force_from_field(GwyDataField *hz, GwyDataField *fz,
                                  GwyMfmProbeType type, gdouble mtip,
                                  gdouble bx, gdouble by, gdouble length)
{
    GwyDataField *rea, *ima, *reb, *imb, *ftf;

    rea = gwy_data_field_new_alike(hz, TRUE);
    reb = gwy_data_field_new_alike(hz, TRUE);
    ima = gwy_data_field_new_alike(hz, TRUE);
    imb = gwy_data_field_new_alike(hz, TRUE);
    ftf = gwy_data_field_new_alike(hz, TRUE);

    gwy_data_field_copy(hz, rea, FALSE);

    gwy_data_field_2dfft_raw(rea, NULL, reb, imb,
                             GWY_TRANSFORM_DIRECTION_FORWARD);

    gwy_data_field_2dfft_humanize(reb);
    gwy_data_field_2dfft_humanize(imb);

    mfm_perpendicular_create_ftf(ftf, mtip,
                             bx, by, length, type);

    gwy_data_field_multiply_fields(reb, reb, ftf);
    gwy_data_field_multiply_fields(imb, imb, ftf);

    gwy_data_field_2dfft_dehumanize(reb);
    gwy_data_field_2dfft_dehumanize(imb);
    gwy_data_field_2dfft_raw(reb, imb, rea, ima,
                             GWY_TRANSFORM_DIRECTION_BACKWARD);


    gwy_data_field_copy(rea, fz, FALSE);

    gwy_object_unref(rea);
    gwy_object_unref(reb);
    gwy_object_unref(ima);
    gwy_object_unref(imb);
    gwy_object_unref(ftf);

}

G_GNUC_UNUSED
static void
mfm_create_ztf(GwyDataField *ztf, gdouble zdiff)
{
    gint i, j;
    gdouble kx, ky, k, *data;

    gint xres = gwy_data_field_get_xres(ztf);
    gint yres = gwy_data_field_get_yres(ztf);
    gdouble xreal = gwy_data_field_get_xreal(ztf);
    gdouble yreal = gwy_data_field_get_yreal(ztf);

    data = gwy_data_field_get_data(ztf);

    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {
            kx = ABS(i - xres/2)/xreal;
            ky = ABS(j - yres/2)/yreal;

            k = hypot(kx, ky);

            data[i*xres + j] = exp(k*zdiff);
        }
    }

}

G_GNUC_UNUSED
static void
mfm_shift_z(GwyDataField *dfield, GwyDataField *out, gdouble zdiff)
{

    GwyDataField *rea, *ima, *reb, *imb, *ztf;

    rea = gwy_data_field_new_alike(dfield, TRUE);
    reb = gwy_data_field_new_alike(dfield, TRUE);
    ima = gwy_data_field_new_alike(dfield, TRUE);
    imb = gwy_data_field_new_alike(dfield, TRUE);
    ztf = gwy_data_field_new_alike(dfield, TRUE);

    gwy_data_field_copy(dfield, rea, FALSE);

    gwy_data_field_2dfft_raw(rea, NULL, reb, imb,
                             GWY_TRANSFORM_DIRECTION_FORWARD);

    gwy_data_field_2dfft_humanize(reb);
    gwy_data_field_2dfft_humanize(imb);

    mfm_create_ztf(ztf, zdiff);

    gwy_data_field_multiply_fields(reb, reb, ztf);
    gwy_data_field_multiply_fields(imb, imb, ztf);

    gwy_data_field_2dfft_dehumanize(reb);
    gwy_data_field_2dfft_dehumanize(imb);
    gwy_data_field_2dfft_raw(reb, imb, rea, ima,
                             GWY_TRANSFORM_DIRECTION_BACKWARD);


    gwy_data_field_copy(rea, out, FALSE);

    gwy_object_unref(rea);
    gwy_object_unref(reb);
    gwy_object_unref(ima);
    gwy_object_unref(imb);
    gwy_object_unref(ztf);
}

G_GNUC_UNUSED
static void
mfm_parallel_create_h(GwyDataField *h, gdouble height, gdouble size_a,
                      gdouble size_b, gdouble size_c, gdouble magnetisation,
                      gdouble thickness, GwyMfmComponentType component)
{
    gint i, j, k;
    gdouble x, pos, *data, val;

    gint *xlist, *dirlist, nlist;

    gint xres = gwy_data_field_get_xres(h);
    gint yres = gwy_data_field_get_yres(h);
    gdouble xreal = gwy_data_field_get_xreal(h);

    xlist = g_new(gint, 100*xres);
    dirlist = g_new(gint, 100*xres);

    nlist = 0;
    pos = -20*(size_a+size_b+thickness+height);

    xlist[nlist] = pos*(gdouble)xres/xreal;
    dirlist[nlist] = 1;
    nlist++;

    do {
       pos += size_a + size_c/2;
       xlist[nlist] = pos*(gdouble)xres/xreal;
       dirlist[nlist] = -1;
       nlist++;

       pos += size_b + size_c/2;
       xlist[nlist] = pos*(gdouble)xres/xreal;
       dirlist[nlist] = 1;
       nlist++;

    } while (pos < xreal+20*(size_a+size_b+thickness+height)
             && nlist < 10*xres-1);

    data = gwy_data_field_get_data(h);

    for (k = 0; k < nlist; k++) {
        for (j = 0; j < xres; j++) {

            x = ((gdouble)j - xlist[k])*xreal/(gdouble)xres;

            if (component == GWY_MFM_COMPONENT_HX)
                val = -dirlist[k]*4*magnetisation
                     *(atan((x*(thickness+height))/(x*x + size_c*size_c + size_c*(thickness+height)))
                     -atan((x*height)/(x*x + size_c*size_c + size_c*height)));
            else if (component == GWY_MFM_COMPONENT_HY)
                val = 0;
            else if (component == GWY_MFM_COMPONENT_HZ)
                val = dirlist[k]*2*magnetisation
                     *log((x*x + (size_c + height + thickness)*(size_c + height + thickness))
                             /(x*x + (size_c + height)*(size_c + height)));
            else {
                g_return_if_reached();
            }

            for (i = 0; i < yres; i++) {
                data[i*xres + j] += val;
            }
        }
    }

    g_free(xlist);
    g_free(dirlist);
}

G_GNUC_UNUSED
static void
mfm_current_create_h(GwyDataField *h, gdouble height, gdouble width,
                     gdouble position, gdouble current,
                     GwyMfmComponentType component)
{
    gint i, j;
    gdouble x, *data, val;
    gint xres = gwy_data_field_get_xres(h);
    gint yres = gwy_data_field_get_yres(h);
    gdouble xreal = gwy_data_field_get_xreal(h);

    data = gwy_data_field_get_data(h);

    for (j = 0; j < xres; j++) {
        x = j*xreal/xres - position;

        if (component == GWY_MFM_COMPONENT_HX)
            val = -current/(2*G_PI*width)*(atan((x-width/2)/height) - atan((x+width/2)/height));
        else if (component == GWY_MFM_COMPONENT_HY)
            val = 0;
        else if (component == GWY_MFM_COMPONENT_HZ)
            val = current/(2*G_PI*width)*log(fabs(cos(atan((x+width/2)/height))/cos(atan((x-width/2)/height))));
        else {
            g_return_if_reached();
        }

        for (i = 0; i < yres; i++) {
            data[i*xres + j] += val;
        }
    }
}

G_GNUC_UNUSED
static void
set_transfer_function_units(GwyDataField *source, GwyDataField *image,
                            GwyDataField *transferfunc)
{
    GwySIUnit *sunit, *iunit, *tunit, *xyunit;

    xyunit = gwy_data_field_get_si_unit_xy(image);
    sunit = gwy_data_field_get_si_unit_z(source);
    iunit = gwy_data_field_get_si_unit_z(image);
    tunit = gwy_data_field_get_si_unit_z(transferfunc);
    gwy_si_unit_divide(iunit, sunit, tunit);
    gwy_si_unit_power_multiply(tunit, 1, xyunit, -2, tunit);
}

#endif

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
