;;; Guile-Git --- GNU Guile bindings of libgit2
;;; Copyright © 2025 Romain Garbage <romain.garbage@inria.fr>
;;;
;;; This file is part of Guile-Git.
;;;
;;; Guile-Git is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; Guile-Git is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with Guile-Git.  If not, see <http://www.gnu.org/licenses/>.

(define-module (tests strarray)
  #:use-module (srfi srfi-64)
  #:use-module (tests helpers)
  #:use-module (system foreign)
  #:use-module (git)
  #:use-module (git structs)
  #:use-module (rnrs bytevectors)
  #:use-module (bytestructures guile))

(test-begin "strarray")

(define string-list '("aaaa" "bbbbb" "cccccc"))

(define string-number-list '("aaaa" "bbbbb" "cccccc" 12))

(define number-list '(1 2 3 4))

(define (pointer->string-list pointer string-count)
  (if (null-pointer? pointer)
      '()
      (map (compose pointer->string make-pointer)
           (pk 'list (bytevector->uint-list
                      (pk 'bv (pointer->bytevector pointer
                                                   (* (sizeof '*)
                                                      string-count)))
                      (native-endianness)
                      (sizeof '*))))))

(define (strarray->string-count-list strarray)
  "Return a list containing the count and the strings (as a string list)
fields of an strarray, in that order."
  (let* ((bytestructure ((@@ (git structs) strarray-bytestructure) strarray))
         (strings (pk 'strings (bytestructure-ref bytestructure 'strings)))
         (count (bytestructure-ref bytestructure 'count)))
    (cons count (pointer->string-list (make-pointer strings) count))))

(test-equal "string-list->strarray: empty list"
  '(0)
  (strarray->string-count-list
   (string-list->strarray '())))

(test-equal "string-list->strarray: string list"
  (cons (length string-list) string-list)
  (strarray->string-count-list
   (string-list->strarray string-list)))

(test-equal "string-list->strarray: list containing a non-string element"
  "Wrong type argument in position 1 (expecting string): 12"
  (catch 'wrong-type-arg
    (lambda ()
     (string-list->strarray string-number-list))
    (lambda (key error fmt args . rest)
      (apply format #f fmt args))))

(test-equal "string-list->strarray: list of non-string elements"
  "Wrong type argument in position 1 (expecting string): 1"
  (catch 'wrong-type-arg
    (lambda ()
      (string-list->strarray number-list))
    (lambda (key error fmt args . rest)
      (apply format #f fmt args))))

(test-end)
