/* Ergo, version 3.3, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2013 Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Kohn−Sham Density Functional Theory Electronic Structure Calculations 
 * with Linearly Scaling Computational Time and Memory Usage,
 * Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek,
 * J. Chem. Theory Comput. 7, 340 (2011),
 * <http://dx.doi.org/10.1021/ct100611z>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */
#ifndef DISTR_BASIS_SET_CHUNK_HEADER
#define DISTR_BASIS_SET_CHUNK_HEADER

#ifdef USE_CHUNKS_AND_TASKS

#include "chunks_and_tasks.h"
#include "basisinfo.h"

struct BoxStruct {
  ergo_real minCoord[3];
  ergo_real maxCoord[3];
  BoxStruct() { 
    for(int i = 0; i < 3; i++) {
      minCoord[i] = 0;
      maxCoord[i] = 0;
    }
  }
};

class DistrBasisSetChunk : public cht::Chunk {
 public:
  // Things required for a Chunk object
  void writeToBuffer ( char * dataBuffer, 
		       size_t const bufferSize ) const;
  size_t getSize() const;
  void assignFromBuffer ( char const * dataBuffer, 
			  size_t const bufferSize);
  size_t memoryUsage() const;
  void getChildChunks(std::list<cht::ChunkID> & childChunkIDs) const;
  // Other stuff
 DistrBasisSetChunk() : noOfBasisFuncs(0) { }
 DistrBasisSetChunk(const BasisInfoStruct & b, const BoxStruct & box) 
   : noOfBasisFuncs(b.noOfBasisFuncs), basisInfo(b), maxExtent(0), boundingBoxForCenters(box) { }
 DistrBasisSetChunk(const BasisInfoStruct & b, const BoxStruct & box, const std::vector<int> indexList) 
   : noOfBasisFuncs(b.noOfBasisFuncs), basisInfo(b), maxExtent(0), boundingBoxForCenters(box), basisFuncIndexList(indexList) { }
 DistrBasisSetChunk(const BasisInfoStruct & b, const BoxStruct & box, const std::vector<int> indexList, const std::vector<ergo_real> extentList) 
   : noOfBasisFuncs(b.noOfBasisFuncs), basisInfo(b), boundingBoxForCenters(box), basisFuncIndexList(indexList), basisFuncExtentList(extentList) {
    maxExtent = 0;
    for(int i = 0; i < basisFuncExtentList.size(); i++)
      if(basisFuncExtentList[i] > maxExtent)
	maxExtent = basisFuncExtentList[i];
  }
 DistrBasisSetChunk(int nBasisFuncs, const BoxStruct & box, ergo_real maxExtent_, cht::ChunkID cid_child1, cht::ChunkID cid_child2) 
   : noOfBasisFuncs(nBasisFuncs), boundingBoxForCenters(box), maxExtent(maxExtent_) {
    cid_child_chunks[0] = cid_child1;
    cid_child_chunks[1] = cid_child2;
  }
  // Data
  int noOfBasisFuncs;
  BasisInfoStruct basisInfo; // Used only at lowest level.
  std::vector<int> basisFuncIndexList; // Used only at lowest level.
  std::vector<ergo_real> basisFuncExtentList; // Used only at lowest level.
  ergo_real maxExtent;
  BoxStruct boundingBoxForCenters;
  cht::ChunkID cid_child_chunks[2]; // Used if not lowest level.
  // Mandatory chunk type declaration macro
  CHT_CHUNK_TYPE_DECLARATION;
};

#endif
#endif
