#include "TransferManager.h"
#include "ClientNotifier.h"
#include "SessionManager.h"
#include <dcpp/QueueManager.h>
#include <dcpp/ConnectionManager.h>
#include "Selecter.h"
#include "CommonTypes.h"
#include "BackendUtil.h"
#include <iostream>
#include <boost/thread/recursive_mutex.hpp>

using namespace std;

namespace dcqt_backend {

TransferManager::TransferManager() 
{
    UploadManager::getInstance()->addListener(this);
    DownloadManager::getInstance()->addListener(this);
    QueueManager::getInstance()->addListener(this);
    FinishedManager::getInstance()->addListener(this);
    if( SETTING(CONNECTION_TYPE) ==  SettingsManager::CONNECTION_ACTIVE ) {
        ConnectionManager::getInstance()->setPort( (short)SETTING(IN_PORT));
        Selecter::doit(ConnectionManager::getInstance()->getServerSocket());
    }
}

TransferManager::~TransferManager()
{
	UploadManager::getInstance()->removeListener(this);
	DownloadManager::getInstance()->removeListener(this);
	QueueManager::getInstance()->removeListener(this);
	FinishedManager::getInstance()->removeListener(this);
	Selecter::quit();
}

void TransferManager::downloadFile(const string& file,int64_t size,int userid, const string& dir, const string& tth) 
{
    ::User::Ptr user = SessionManager::instance()->getUserIdMap()[userid];

    // The user pointer could be 0 because it has been released.
    // TODO we could really make use of a return status value here
    if(!user) {
      return;
    }
	// If dir is empty we use the default directory
	string target = !dir.empty() ? dir : SETTING (DOWNLOAD_DIRECTORY);
	
    target += Util::getFileName(BackendUtil::linuxSeparator (file));
    // HERE we COULD catch an exception and return false and propagate that back to the GUI.

    try {
        if ( size == -1 ) {
	string winFile = BackendUtil::windowsSeparator(file);
			cout << "#################" << endl << 
				"File: " << winFile.c_str() << 
				" Nick: " << user->getNick().c_str() <<
				" Target " << target.c_str() << endl;
            QueueManager::getInstance()->addDirectory( BackendUtil::windowsSeparator(file), user, target );
        } else {
          TTHValue tthvalue(tth);
	  QueueManager::getInstance()->add
            ( file, size, user, target, &tthvalue, QueueItem::FLAG_RESUME);// | QueueItem::FLAG_SOURCE_UTF8);
        }
    } 
	catch( const QueueException& qe ) {
	}
	catch( const FileException& fe ) {
	}
}

void TransferManager::getUserFileList( int userId ) 
{
	try {
		
		::User::Ptr user = SessionManager::instance()->getUserIdMap()[userId];
		
		if ( user ) {
        	QueueManager::getInstance()->addList(user, QueueItem::FLAG_NORMAL);
        	fileListRequests.push_back( userId );
    	} else {
	    }
	}
	catch( const QueueException& qe ) {
	}
	catch( const FileException& fe ) {
	}
}

void TransferManager::forceConnect(int userId)
{
	boost::recursive_mutex::scoped_lock sl(SessionManager::instance()->sessionLock);
  //SessionManager::instance()->aquireLock();
  ::User::Ptr user = SessionManager::instance()->getUserIdMap()[userId];
  try {
    if(user) user->connect();
  } catch (...) {}
  //SessionManager::instance()->releaseLock();
}

void TransferManager::on(UploadManagerListener::Starting, Upload* u) throw() 
{
    uploadStatus[u] = UPLOAD_STARTING; //TODO t3mp solution, mem leaks and no throttling
    ClientNotifier::instance()->transferEvent( ClientNotifier::STARTING, u );
}

void TransferManager::on(UploadManagerListener::Tick, const Upload::List& ul) throw() 
{
    ClientNotifier::instance()->transferTick( ul,uploadStatus );
}

void TransferManager::on(UploadManagerListener::Complete, Upload* u) throw() 
{
    uploadStatus[u] = UPLOAD_COMPLETE;
    ClientNotifier::instance()->transferEvent( ClientNotifier::COMPLETE, u );
}

void TransferManager::on(UploadManagerListener::Failed, Upload* u, const string& s) throw() {
	
    uploadStatus[u] = UPLOAD_COMPLETE;
    ClientNotifier::instance()->transferEvent( u, s );
}

void TransferManager::on(DownloadManagerListener::Starting, Download* d) throw() 
{
    ClientNotifier::instance()->transferEvent( ClientNotifier::STARTING, d );
}

void TransferManager::on(DownloadManagerListener::Tick, const Download::List& dl) throw() 
{
    ClientNotifier::instance()->transferTick( dl );
}


void TransferManager::on(DownloadManagerListener::Complete, Download* d) throw() 
{
    ClientNotifier::instance()->transferEvent( ClientNotifier::COMPLETE, d );
}


void TransferManager::on(DownloadManagerListener::Failed, Download* d, const string& s) throw() 
{
    ClientNotifier::instance()->transferEvent( d, s );
}

void TransferManager::on(QueueManagerListener::Added, QueueItem* item) throw() 
{
    int qid = queueItemIds.getId(item);
    ClientNotifier::instance()->queueEvent( ClientNotifier::QUEUE_ADD, item, qid);
}

void TransferManager::on(QueueManagerListener::Finished, QueueItem* item) throw() 
{
    int qid = queueItemIds.getId(item);
    ClientNotifier::instance()->queueEvent( ClientNotifier::QUEUE_FINISH, item, qid);
}

void TransferManager::on(QueueManagerListener::Removed, QueueItem* item) throw() 
{
    int qid = queueItemIds.getId(item);
    ClientNotifier::instance()->queueEvent( ClientNotifier::QUEUE_REMOVE, item, qid);
    queueItemIds.release(item);
}

void TransferManager::on(QueueManagerListener::Moved, QueueItem*) throw() 
{
}

void TransferManager::on(QueueManagerListener::SourcesUpdated, QueueItem* item) throw() 
{
    int qid = queueItemIds.getId(item);
    ClientNotifier::instance()->queueEvent( ClientNotifier::QUEUE_SOURCE_UPDATE, item, qid);
}

void TransferManager::on(QueueManagerListener::StatusUpdated, QueueItem* item) throw() 
{
    int qid = queueItemIds.getId(item);
    ClientNotifier::instance()->queueEvent( ClientNotifier::QUEUE_STATUS_UPDATE, item, qid);
}

void TransferManager::on(QueueManagerListener::SearchStringUpdated, QueueItem*) throw() 
{
}

void TransferManager::on(QueueManagerListener::PartialList, const ::User::Ptr&, const string&) throw() 
{
}


void TransferManager::on(FinishedManagerListener::AddedDl, FinishedItem* item) throw() 
{
    int id = finishedItemIds.getId( item );
    ClientNotifier::instance()->finishedEvent(DOWNLOAD,id,item);
}

void TransferManager::on(FinishedManagerListener::RemovedDl, FinishedItem* item) throw() 
{
    int id = finishedItemIds.getId( item );
    ClientNotifier::instance()->finishedEvent(DOWNLOAD,id);
    finishedItemIds.release(item);
}

void TransferManager::on(FinishedManagerListener::RemovedAllDl) throw() 
{
    ClientNotifier::instance()->finishedEvent(DOWNLOAD);
}

void TransferManager::on(FinishedManagerListener::AddedUl, FinishedItem* item) throw() 
{
    int id = finishedItemIds.getId( item );
    ClientNotifier::instance()->finishedEvent(UPLOAD,id,item);
}

void TransferManager::on(FinishedManagerListener::RemovedUl, FinishedItem* item) throw() 
{
    int id = finishedItemIds.getId( item );
    ClientNotifier::instance()->finishedEvent(UPLOAD,id);
    finishedItemIds.release(item);
}

void TransferManager::on(FinishedManagerListener::RemovedAllUl) throw() 
{
    ClientNotifier::instance()->finishedEvent(UPLOAD);
}


bool TransferManager::isFileListRequested( int uId ) {
    bool ret = false;
    vector<int>::const_iterator it = fileListRequests.begin();
	
    while( it != fileListRequests.end() ) {
        if( *it == uId ) {
            ret = true;
            break;
        }
        it++;
    }
	
    return ret;
}

void TransferManager::removeFileListRequest( int uId ) 
{
	
    vector<int>::iterator it = fileListRequests.begin();
	
    // Loop through the whole list of requests
    while( it != fileListRequests.end() ) {
        if( *it == uId ) {
            // Remove the item from the list
            it = fileListRequests.erase(it);
        } else {
            it++;
        }
    }
}

void TransferManager::removeQueueItem(int qid)
{
  // find ze item
  if( queueItemIds.contains(qid) ) {
    QueueItem::Ptr qptr = queueItemIds.get(qid);
    QueueManager::getInstance()->remove(qptr->getTarget());
  }
}

void TransferManager::removeSource(int qid, int uid)
{
	boost::recursive_mutex::scoped_lock sl(SessionManager::instance()->sessionLock);
	if( queueItemIds.contains(qid) )
	{
		// Aquire the QueueItem
		QueueItem::Ptr qptr = queueItemIds.get(qid);
		// Aquire the User
		::User::Ptr uptr = SessionManager::instance()->getUserIdMap()[uid];
		// Remove
		QueueManager::getInstance()->removeSource(qptr->getTarget(),uptr,QueueItem::Source::FLAG_REMOVED);
	}	
}


}
