/*
 * Original work Copyright 2009 - 2010 Kevin Ackley (kackley@gwi.net)
 * Modified work Copyright 2018 - 2020 Andy Maloney <asmaloney@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person or organization
 * obtaining a copy of the software and accompanying documentation covered by
 * this license (the "Software") to use, reproduce, display, distribute,
 * execute, and transmit the Software, and to prepare derivative works of the
 * Software, and to permit third-parties to whom the Software is furnished to
 * do so, all subject to the following:
 *
 * The copyright notices in the Software and this entire statement, including
 * the above license grant, this restriction and the following disclaimer,
 * must be included in all copies of the Software, in whole or in part, and
 * all derivative works of the Software, unless such copies or derivative
 * works are solely in the form of machine-executable object code generated by
 * a source language processor.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
 * FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "DecodeChannel.h"
#include "SourceDestBufferImpl.h"
#include "StringFunctions.h"

namespace e57
{
   DecodeChannel::DecodeChannel( SourceDestBuffer dbuf_arg, std::shared_ptr<Decoder> decoder_arg,
                                 unsigned bytestreamNumber_arg, uint64_t maxRecordCount_arg ) :
      dbuf( dbuf_arg ), decoder( decoder_arg ), bytestreamNumber( bytestreamNumber_arg )
   {
      maxRecordCount = maxRecordCount_arg;
      currentPacketLogicalOffset = 0;
      currentBytestreamBufferIndex = 0;
      currentBytestreamBufferLength = 0;
      inputFinished = false;
   }

   bool DecodeChannel::isOutputBlocked() const
   {
      // If we have completed the entire vector, we are done
      if ( decoder->totalRecordsCompleted() >= maxRecordCount )
      {
         return ( true );
      }

      // If we have filled the dest buffer, we are blocked
      return ( dbuf.impl()->nextIndex() == dbuf.impl()->capacity() );
   }

   bool DecodeChannel::isInputBlocked() const
   {
      // If have read until the section end, we are done
      if ( inputFinished )
      {
         return ( true );
      }

      // If have eaten all the input in the current packet, we are blocked.
      return ( currentBytestreamBufferIndex == currentBytestreamBufferLength );
   }

#ifdef E57_ENABLE_DIAGNOSTIC_OUTPUT
   void DecodeChannel::dump( int indent, std::ostream &os ) const
   {
      os << space( indent ) << "dbuf" << std::endl;
      dbuf.dump( indent + 4, os );

      os << space( indent ) << "decoder:" << std::endl;
      decoder->dump( indent + 4, os );

      os << space( indent ) << "bytestreamNumber:              " << bytestreamNumber << std::endl;
      os << space( indent ) << "maxRecordCount:                " << maxRecordCount << std::endl;
      os << space( indent ) << "currentPacketLogicalOffset:    " << currentPacketLogicalOffset
         << std::endl;
      os << space( indent ) << "currentBytestreamBufferIndex:  " << currentBytestreamBufferIndex
         << std::endl;
      os << space( indent ) << "currentBytestreamBufferLength: " << currentBytestreamBufferLength
         << std::endl;
      os << space( indent ) << "inputFinished:                 " << inputFinished << std::endl;
      os << space( indent ) << "isInputBlocked():              " << isInputBlocked() << std::endl;
      os << space( indent ) << "isOutputBlocked():             " << isOutputBlocked() << std::endl;
   }
#endif
}
