use std::{
    fmt::Display,
    sync::{
        atomic::{AtomicBool, Ordering},
        Arc,
    },
    time::{Duration, Instant},
};

use log::{debug, info, warn};
use logind_zbus::{
    manager::{ManagerProxy, SessionInfo},
    session::{SessionClass, SessionProxy, SessionState, SessionType},
};
use serde::{Deserialize, Serialize};
use tokio::time::sleep;
use zbus::zvariant::Type;
use zbus::Connection;

use crate::{
    config::{check_vulkan_icd, create_modprobe_conf, GfxConfig},
    do_driver_action,
    error::GfxError,
    kill_nvidia_lsof,
    pci_device::{rescan_pci_bus, DiscreetGpu, GfxMode, GfxVendor, HotplugState, HotplugType},
    special_asus::{asus_dgpu_set_disabled, asus_egpu_set_enabled, asus_gpu_mux_set_igpu},
    systemd::{
        do_systemd_unit_action, wait_systemd_unit_state, SystemdUnitAction, SystemdUnitState,
    },
    toggle_nvidia_powerd, DriverAction, DISPLAY_MANAGER, VFIO_DRIVERS,
};

pub enum Action {
    UserAction(UserActionRequired),
    StagedActions(Vec<StagedAction>),
}

#[derive(Debug, Clone, Copy, Serialize, Deserialize, Type)]
/// The action required by the user after they request a supergfx action
pub enum UserActionRequired {
    Logout,
    Reboot,
    SwitchToIntegrated,
    AsusEgpuDisable,
    Nothing,
}

impl UserActionRequired {
    /// Determine if we need to logout/thread. Integrated<->Vfio mode does not
    /// require logout.
    pub fn mode_change_action(new_mode: GfxMode, current_mode: GfxMode) -> Self {
        match new_mode {
            GfxMode::Hybrid => match current_mode {
                GfxMode::Integrated | GfxMode::AsusEgpu => Self::Logout,
                GfxMode::AsusMuxDgpu => Self::Reboot,
                GfxMode::Vfio => Self::SwitchToIntegrated,
                GfxMode::NvidiaNoModeset | GfxMode::Hybrid | GfxMode::None => Self::Nothing,
            },
            GfxMode::Integrated => match current_mode {
                GfxMode::Hybrid | GfxMode::AsusEgpu => Self::Logout,
                GfxMode::AsusMuxDgpu => Self::Reboot,
                GfxMode::Vfio | GfxMode::NvidiaNoModeset | GfxMode::Integrated | GfxMode::None => {
                    Self::Nothing
                }
            },
            GfxMode::NvidiaNoModeset => match current_mode {
                GfxMode::Integrated
                | GfxMode::NvidiaNoModeset
                | GfxMode::Vfio
                | GfxMode::Hybrid
                | GfxMode::None => Self::Nothing,
                GfxMode::AsusEgpu => Self::Logout,
                GfxMode::AsusMuxDgpu => Self::Reboot,
            },
            GfxMode::Vfio => match current_mode {
                GfxMode::Integrated | GfxMode::Vfio | GfxMode::NvidiaNoModeset | GfxMode::None => {
                    Self::Nothing
                }
                GfxMode::AsusEgpu | GfxMode::Hybrid => Self::Logout,
                GfxMode::AsusMuxDgpu => Self::Reboot,
            },
            GfxMode::AsusEgpu => match current_mode {
                GfxMode::Integrated | GfxMode::Hybrid | GfxMode::NvidiaNoModeset => Self::Logout,
                GfxMode::Vfio => Self::SwitchToIntegrated,
                GfxMode::AsusEgpu | GfxMode::None => Self::Nothing,
                GfxMode::AsusMuxDgpu => Self::Reboot,
            },
            GfxMode::AsusMuxDgpu => match current_mode {
                GfxMode::Hybrid
                | GfxMode::Integrated
                | GfxMode::NvidiaNoModeset
                | GfxMode::Vfio
                | GfxMode::AsusEgpu => Self::Reboot,
                GfxMode::None | GfxMode::AsusMuxDgpu => Self::Nothing,
            },
            GfxMode::None => Self::Nothing,
        }
    }
}

impl Display for UserActionRequired {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            Self::Logout => write!(f, "Logout"),
            Self::Reboot => write!(f, "Reboot"),
            Self::SwitchToIntegrated => write!(f, "SwitchToIntegrated"),
            Self::AsusEgpuDisable => write!(f, "AsusEgpuDisable"),
            Self::Nothing => write!(f, "Nothing"),
        }
    }
}

impl From<UserActionRequired> for &str {
    /// Convert the action to a verbose string
    fn from(gfx: UserActionRequired) -> &'static str {
        match gfx {
            UserActionRequired::Logout => "Logout required to complete mode change",
            UserActionRequired::Reboot => "Reboot required to complete mode change",
            UserActionRequired::SwitchToIntegrated => "You must switch to Integrated first",
            UserActionRequired::Nothing => "No action required",
            UserActionRequired::AsusEgpuDisable => {
                "The mode must be switched to Integrated or Hybrid first"
            }
        }
    }
}

impl From<&UserActionRequired> for &str {
    fn from(gfx: &UserActionRequired) -> &'static str {
        (*gfx).into()
    }
}

/// All the possible actions supergfx can perform. These should be chucked in
/// a vector in the order required to perform them.
#[derive(Debug, PartialEq, PartialOrd, Clone, Copy)]
pub enum StagedAction {
    /// Wait for the user to logout
    WaitLogout,
    /// Stop the display manager
    StopDisplayManager,
    /// Restart the display manager
    StartDisplayManager,
    /// A marker for no logind options
    NoLogind,
    /// Load the dgpu drivers
    LoadGpuDrivers,
    /// Unload the dgpu drivers
    UnloadGpuDrivers,
    /// Kill all things using the nvidia device
    KillNvidia,
    /// Kill all things using the AMD device
    KillAmd,
    /// Enable nvidia-powerd service
    EnableNvidiaPowerd,
    /// Disable nvidia-powerd service
    DisableNvidiaPowerd,
    /// Load the vfio modules
    LoadVfioDrivers,
    /// Unload the vfio modules
    UnloadVfioDrivers,
    /// A none-action marker to specify an intent, in this case not using ASUS or hotplug device removal and only dev-tree unbind/remove
    DevTreeManaged,
    RescanPci,
    /// Unbind and fully remove the device from a driver using sysfs
    UnbindRemoveGpu,
    /// Unbind only, device is still in PCI tree
    UnbindGpu,
    /// If hotplug is available then the dgpu can be hot-removed
    HotplugUnplug,
    /// If hotplug is available then the dgpu can be hot-plugged
    HotplugPlug,
    /// Disable the internal dgpu using the ASUS ACPI method. This does a hard removal of the device and a pci-scan will no-longer find it
    AsusDgpuDisable,
    /// Enable the internal dgpu using the ASUS ACPI method. This must be done for it to be seen on the pci bus again after an `AsusDgpuDisable`
    AsusDgpuEnable,
    /// This will also disable the internal dgpu due to the laptop ACPI functions being called
    AsusEgpuDisable,
    /// This will also enable the internal dgpu due to the laptop ACPI functions being called
    AsusEgpuEnable,
    /// Switch the ASUS MUX to igpu mode
    AsusMuxIgpu,
    /// Switch the ASUS MUX to dgpu mode
    AsusMuxDgpu,
    /// Write a modprobe conf according to mode (e.g, hybrid, vfio)
    WriteModprobeConf,
    /// Checks for correct Vulkan ICD (remove nvidia_icd.json if not on "nvidia" or "vfio")
    CheckVulkanIcd,
    /// Placeholder, used to indicate the dgpu is not Nvidia (for example when deciding if KillNvidia should be used)
    NotNvidia,
    None,
}

impl StagedAction {
    /// Generate a series of initial mode steps, these are specific to booting the system only, not changing modes
    pub fn action_list_for_boot(
        config: &GfxConfig,
        vendor: GfxVendor,
        mode: GfxMode,
    ) -> Vec<StagedAction> {
        let kill_gpu_use = if vendor == GfxVendor::Nvidia {
            Self::KillNvidia
        } else {
            Self::KillAmd
        };

        let disable_nvidia_powerd = if vendor == GfxVendor::Nvidia {
            Self::DisableNvidiaPowerd
        } else {
            Self::NotNvidia
        };

        let enable_nvidia_powerd = if vendor == GfxVendor::Nvidia {
            Self::EnableNvidiaPowerd
        } else {
            Self::NotNvidia
        };

        let hotplug_rm_type = match config.hotplug_type {
            HotplugType::Std => Self::HotplugUnplug,
            HotplugType::Asus => Self::AsusDgpuDisable,
            HotplugType::None => Self::DevTreeManaged,
        };

        let hotplug_add_type = match config.hotplug_type {
            HotplugType::Std => Self::HotplugPlug,
            HotplugType::Asus => Self::AsusDgpuEnable,
            HotplugType::None => Self::DevTreeManaged,
        };

        match mode {
            GfxMode::Hybrid => vec![
                Self::WriteModprobeConf,
                Self::CheckVulkanIcd,
                hotplug_add_type,
                Self::RescanPci,
                Self::LoadGpuDrivers,
                enable_nvidia_powerd,
            ],
            GfxMode::Integrated | GfxMode::NvidiaNoModeset => vec![
                disable_nvidia_powerd,
                kill_gpu_use,
                Self::UnloadGpuDrivers,
                Self::UnbindRemoveGpu,
                Self::WriteModprobeConf,
                Self::CheckVulkanIcd,
                hotplug_rm_type,
            ],
            GfxMode::Vfio => vec![
                disable_nvidia_powerd,
                kill_gpu_use,
                Self::UnloadGpuDrivers,
                Self::WriteModprobeConf,
                Self::CheckVulkanIcd,
                Self::LoadVfioDrivers,
            ],
            GfxMode::AsusEgpu => vec![
                Self::WriteModprobeConf,
                Self::CheckVulkanIcd,
                Self::LoadGpuDrivers,
                enable_nvidia_powerd,
            ],
            GfxMode::AsusMuxDgpu => vec![
                // TODO: remove iGPU
                Self::WriteModprobeConf,
                Self::CheckVulkanIcd,
                Self::LoadGpuDrivers,
                enable_nvidia_powerd,
            ],
            GfxMode::None => vec![],
        }
    }

    /// Generate a well defined list of specific actions required for the mode switch.
    //
    // There might be some redundancy in this list but it is preferred so as to force checking of all conditions for from/to combos
    pub fn action_list_for_switch(
        config: &GfxConfig,
        vendor: GfxVendor,
        from: GfxMode,
        to: GfxMode,
    ) -> Action {
        let mut wait_logout = Self::NoLogind;
        let mut stop_display = Self::NoLogind;
        let mut start_display = Self::NoLogind;
        if !config.no_logind & !config.always_reboot {
            wait_logout = Self::WaitLogout;
            stop_display = Self::StopDisplayManager;
            start_display = Self::StartDisplayManager;
        };

        let mut kill_gpu_use = Self::NotNvidia;
        // the nvida powerd toggle function only runs if the vendor is nvidia
        let disable_nvidia_powerd = Self::DisableNvidiaPowerd;
        let enable_nvidia_powerd = Self::EnableNvidiaPowerd;
        if vendor == GfxVendor::Nvidia {
            kill_gpu_use = Self::KillNvidia;
            // disable_nvidia_powerd = Self::DisableNvidiaPowerd;
            // enable_nvidia_powerd = Self::EnableNvidiaPowerd;
        } else if vendor == GfxVendor::Amd {
            kill_gpu_use = Self::KillAmd;
        }

        let hotplug_rm_type = match config.hotplug_type {
            HotplugType::Std => Self::HotplugUnplug,
            HotplugType::Asus => Self::AsusDgpuDisable,
            HotplugType::None => Self::DevTreeManaged,
        };

        let hotplug_add_type = match config.hotplug_type {
            HotplugType::Std => Self::HotplugPlug,
            HotplugType::Asus => Self::AsusDgpuEnable,
            HotplugType::None => Self::DevTreeManaged,
        };

        // Be verbose in this list of actions. It's okay to have repeated blocks as this makes it much clearer
        // which action chain results from which switching combo
        match from {
            GfxMode::Hybrid => match to {
                GfxMode::Integrated => Action::StagedActions(vec![
                    wait_logout,
                    stop_display,
                    disable_nvidia_powerd,
                    kill_gpu_use,
                    Self::UnloadGpuDrivers,
                    Self::UnbindRemoveGpu,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    hotplug_rm_type,
                    start_display,
                ]),
                // Ask the user to do the switch instead of doing something unexpected
                GfxMode::Vfio => Action::UserAction(UserActionRequired::SwitchToIntegrated),
                GfxMode::AsusEgpu => Action::StagedActions(vec![
                    wait_logout,
                    stop_display,
                    disable_nvidia_powerd,
                    kill_gpu_use,
                    Self::UnloadGpuDrivers,
                    Self::UnbindRemoveGpu,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    Self::AsusEgpuEnable,
                    Self::RescanPci,
                    Self::LoadGpuDrivers,
                    enable_nvidia_powerd,
                    start_display,
                ]),
                GfxMode::AsusMuxDgpu => Action::StagedActions(vec![
                    // Self::WriteModprobeConf,
                    Self::CheckVulkanIcd, // check this in anycase
                    enable_nvidia_powerd,
                    Self::AsusMuxDgpu,
                ]),
                GfxMode::Hybrid | GfxMode::NvidiaNoModeset | GfxMode::None => {
                    Action::UserAction(UserActionRequired::Nothing)
                }
            },
            GfxMode::Integrated => match to {
                GfxMode::Hybrid => Action::StagedActions(vec![
                    wait_logout,
                    stop_display,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    hotplug_add_type,
                    Self::RescanPci,
                    Self::LoadGpuDrivers,
                    enable_nvidia_powerd,
                    start_display,
                ]),
                GfxMode::NvidiaNoModeset => Action::StagedActions(vec![
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    Self::RescanPci,
                    Self::LoadGpuDrivers,
                    enable_nvidia_powerd,
                ]),
                GfxMode::Vfio => Action::StagedActions(vec![
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    Self::RescanPci, // Make the PCI devices available
                    disable_nvidia_powerd,
                    kill_gpu_use,
                    Self::UnloadGpuDrivers, // rescan can load the gpu drivers automatically
                    Self::UnbindGpu,
                    Self::LoadVfioDrivers,
                ]),
                GfxMode::AsusEgpu => Action::StagedActions(vec![
                    wait_logout,
                    stop_display,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    Self::AsusEgpuEnable,
                    Self::RescanPci,
                    Self::LoadGpuDrivers,
                    enable_nvidia_powerd,
                    start_display,
                ]),
                GfxMode::AsusMuxDgpu => Action::StagedActions(vec![
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    hotplug_add_type, // must always assume the possibility dgpu_disable was set
                    enable_nvidia_powerd,
                    Self::AsusMuxDgpu,
                ]),
                GfxMode::Integrated | GfxMode::None => {
                    Action::UserAction(UserActionRequired::Nothing)
                }
            },
            GfxMode::NvidiaNoModeset => match to {
                GfxMode::Hybrid => Action::UserAction(UserActionRequired::Nothing),
                GfxMode::Integrated => Action::StagedActions(vec![
                    disable_nvidia_powerd,
                    kill_gpu_use,
                    Self::UnloadGpuDrivers,
                    Self::UnbindRemoveGpu,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                ]),
                GfxMode::Vfio => Action::StagedActions(vec![
                    disable_nvidia_powerd,
                    kill_gpu_use,
                    Self::UnloadGpuDrivers,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    Self::LoadVfioDrivers,
                ]),
                GfxMode::AsusEgpu => Action::UserAction(UserActionRequired::Nothing),
                GfxMode::AsusMuxDgpu => Action::StagedActions(vec![
                    // Self::WriteModprobeConf,
                    enable_nvidia_powerd,
                    Self::AsusMuxDgpu,
                ]),
                GfxMode::NvidiaNoModeset | GfxMode::None => {
                    Action::UserAction(UserActionRequired::Nothing)
                }
            },
            GfxMode::Vfio => match to {
                GfxMode::Hybrid | GfxMode::NvidiaNoModeset => Action::StagedActions(vec![
                    kill_gpu_use,
                    Self::UnloadVfioDrivers,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    Self::RescanPci,
                    Self::LoadGpuDrivers,
                ]),
                GfxMode::Integrated => Action::StagedActions(vec![
                    kill_gpu_use,
                    Self::UnloadVfioDrivers,
                    Self::UnbindRemoveGpu,
                ]),
                GfxMode::AsusEgpu => Action::StagedActions(vec![
                    wait_logout,
                    stop_display,
                    Self::UnloadVfioDrivers,
                    Self::UnbindRemoveGpu,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    Self::AsusEgpuEnable,
                    Self::RescanPci,
                    Self::LoadGpuDrivers,
                    enable_nvidia_powerd,
                    start_display,
                ]),
                GfxMode::AsusMuxDgpu => Action::StagedActions(vec![
                    // Self::WriteModprobeConf,
                    enable_nvidia_powerd,
                    Self::AsusMuxDgpu,
                ]),
                GfxMode::Vfio | GfxMode::None => Action::UserAction(UserActionRequired::Nothing),
            },
            GfxMode::AsusEgpu => match to {
                GfxMode::Hybrid => Action::StagedActions(vec![
                    wait_logout,
                    stop_display,
                    disable_nvidia_powerd,
                    kill_gpu_use,
                    Self::UnloadGpuDrivers,
                    Self::UnbindRemoveGpu,
                    Self::WriteModprobeConf,
                    Self::CheckVulkanIcd,
                    Self::AsusEgpuDisable,
                    Self::AsusDgpuEnable, // ensure the dgpu is enabled
                    Self::RescanPci,
                    Self::LoadGpuDrivers,
                    enable_nvidia_powerd,
                    start_display,
                ]),
                GfxMode::Integrated => Action::StagedActions(vec![
                    wait_logout,
                    stop_display,
                    disable_nvidia_powerd,
                    kill_gpu_use,
                    Self::UnloadGpuDrivers,
                    Self::UnbindRemoveGpu,
                    Self::WriteModprobeConf,
                    Self::AsusEgpuDisable,
                    Self::UnloadGpuDrivers,
                    Self::UnbindRemoveGpu, // egpu disable also enable dgpu, which can reload the drivers
                    Self::WriteModprobeConf, // TODO: called twice? (why?)
                    Self::CheckVulkanIcd,
                    hotplug_rm_type, // also need to ensure dgpu is off
                    start_display,
                ]),
                GfxMode::Vfio => Action::UserAction(UserActionRequired::SwitchToIntegrated),
                GfxMode::AsusMuxDgpu => Action::UserAction(UserActionRequired::AsusEgpuDisable),
                GfxMode::AsusEgpu | GfxMode::NvidiaNoModeset | GfxMode::None => {
                    Action::UserAction(UserActionRequired::Nothing)
                }
            },
            // The mux change *ALWAYS* requires a reboot, so only switch to/from mux and hybrid
            GfxMode::AsusMuxDgpu => match to {
                GfxMode::AsusMuxDgpu => Action::UserAction(UserActionRequired::Nothing),
                _ => Action::StagedActions(vec![Self::AsusMuxIgpu]),
            },
            GfxMode::None => Action::UserAction(UserActionRequired::Nothing),
        }
    }

    /// Do the work required by the action
    pub async fn perform(
        &self,
        changing_to: GfxMode,
        device: &mut DiscreetGpu,
        loop_exit: Arc<AtomicBool>,
    ) -> Result<(), GfxError> {
        match self {
            StagedAction::WaitLogout => wait_logout(loop_exit).await,
            StagedAction::StopDisplayManager => {
                do_systemd_unit_action(SystemdUnitAction::Stop, DISPLAY_MANAGER)?;
                wait_systemd_unit_state(SystemdUnitState::Inactive, DISPLAY_MANAGER)
            }
            StagedAction::StartDisplayManager => {
                do_systemd_unit_action(SystemdUnitAction::Start, DISPLAY_MANAGER)
            }
            StagedAction::LoadGpuDrivers => device.do_driver_action(DriverAction::Load),
            StagedAction::UnloadGpuDrivers => device.do_driver_action(DriverAction::Remove),
            StagedAction::LoadVfioDrivers => do_driver_action("vfio-pci", DriverAction::Load),
            StagedAction::UnloadVfioDrivers => {
                for driver in VFIO_DRIVERS.iter() {
                    do_driver_action(driver, DriverAction::Remove)?;
                }
                Ok(())
            }
            StagedAction::KillNvidia => kill_nvidia_lsof(),
            StagedAction::KillAmd => {
                // TODO: do this
                Ok(())
            }
            StagedAction::EnableNvidiaPowerd => toggle_nvidia_powerd(true, device.vendor()),
            StagedAction::DisableNvidiaPowerd => toggle_nvidia_powerd(false, device.vendor()),
            StagedAction::RescanPci => rescan_pci(device),
            StagedAction::UnbindRemoveGpu => device.unbind_remove(),
            StagedAction::UnbindGpu => device.unbind(),
            StagedAction::HotplugUnplug => device.set_hotplug(HotplugState::Off),
            StagedAction::HotplugPlug => device.set_hotplug(HotplugState::On),
            StagedAction::AsusDgpuDisable => asus_dgpu_set_disabled(true),
            StagedAction::AsusDgpuEnable => asus_dgpu_set_disabled(false),
            StagedAction::AsusEgpuDisable => asus_egpu_set_enabled(false),
            StagedAction::AsusEgpuEnable => asus_egpu_set_enabled(true),
            StagedAction::AsusMuxIgpu => asus_gpu_mux_set_igpu(true),
            StagedAction::AsusMuxDgpu => asus_gpu_mux_set_igpu(false),
            StagedAction::WriteModprobeConf => create_modprobe_conf(changing_to, device),
            StagedAction::CheckVulkanIcd => {
                check_vulkan_icd(changing_to)
                    .map_err(|e| warn!("Vulkan ICD failed: {e:?}"))
                    .ok();
                Ok(())
            }
            StagedAction::DevTreeManaged => Ok(()),
            StagedAction::NoLogind => Ok(()),
            StagedAction::NotNvidia => Ok(()),
            StagedAction::None => Ok(()),
        }
    }
}

/// Check if the user has any graphical uiser sessions that are active or online
async fn graphical_user_sessions_exist(
    connection: &Connection,
    sessions: &[SessionInfo],
) -> Result<bool, GfxError> {
    for session in sessions {
        // should ignore error such as:
        // Zbus error: org.freedesktop.DBus.Error.UnknownObject: Unknown object '/org/freedesktop/login1/session/c2'
        if let Ok(session_proxy) = SessionProxy::builder(connection)
            .path(session.path())?
            .build()
            .await
            .map_err(|e| warn!("graphical_user_sessions_exist: builder: {e:?}"))
        {
            if let Ok(class) = session_proxy.class().await.map_err(|e| {
                warn!("graphical_user_sessions_exist: class: {e:?}");
                e
            }) {
                if class == SessionClass::User {
                    if let Ok(type_) = session_proxy.type_().await.map_err(|e| {
                        warn!("graphical_user_sessions_exist: type_: {e:?}");
                        e
                    }) {
                        match type_ {
                            SessionType::X11 | SessionType::Wayland | SessionType::MIR => {
                                if let Ok(state) = session_proxy.state().await.map_err(|e| {
                                    warn!("graphical_user_sessions_exist: state: {e:?}");
                                    e
                                }) {
                                    match state {
                                        SessionState::Online | SessionState::Active => {
                                            return Ok(true)
                                        }
                                        SessionState::Closing => {}
                                    }
                                }
                            }
                            _ => {}
                        }
                    }
                }
            }
        }
    }
    Ok(false)
}

/// It's async because of inner calls, but is a blocking loop
// TODO: make it a Future
async fn wait_logout(loop_exit: Arc<AtomicBool>) -> Result<(), GfxError> {
    loop_exit.store(false, Ordering::Release);

    const SLEEP_PERIOD: Duration = Duration::from_millis(100);
    let logout_timeout_s = 30;
    let start_time = Instant::now();

    let connection = Connection::system().await?;
    let manager = ManagerProxy::new(&connection).await?;

    while !loop_exit.load(Ordering::Acquire) {
        let sessions = manager.list_sessions().await?;

        if !graphical_user_sessions_exist(&connection, &sessions).await? {
            break;
        }

        // exit if 3 minutes pass
        if logout_timeout_s != 0
            && Instant::now().duration_since(start_time).as_secs() > logout_timeout_s
        {
            let detail = format!("Time ({} seconds) for logout exceeded", logout_timeout_s);
            warn!("mode_change_loop: {}", detail);
            return Err(GfxError::SystemdUnitWaitTimeout(detail));
        }

        // Don't spin at max speed
        sleep(SLEEP_PERIOD).await;
    }

    loop_exit.store(false, Ordering::Release);
    debug!("wait_logout: loop exited");
    Ok(())
}

fn rescan_pci(device: &mut DiscreetGpu) -> Result<(), GfxError> {
    // Don't do a rescan unless the dev list is empty. This might be the case if
    // asus dgpu_disable is set before the daemon starts. But in general the daemon
    // should have the correct device on boot and retain that.
    let mut do_find_device = device.devices().is_empty();
    for dev in device.devices() {
        if dev.is_dgpu() {
            do_find_device = false;
            break;
        }
        do_find_device = true;
    }

    if do_find_device {
        info!("do_rescan: Device rescan required");
        match DiscreetGpu::new() {
            Ok(dev) => *device = dev,
            Err(e) => warn!("do_rescan: tried to reset Unknown dgpu status/devices: {e:?}"),
        }
    } else {
        info!("do_rescan: Rescanning PCI bus");
        rescan_pci_bus()?; // should force re-attach of driver
    }

    Ok(())
}
