/*
 * Copyright (c) 2019-2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sve_hybrid_fp32_mla_8x1VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<float> A_arg,
    size_t M, size_t N, const float *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const float *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const float *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p2.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x8\n"
      "bge 92f\n"
      "cmp %x[M], #0x6\n"
      "bgt 79f\n"
      "beq 66f\n"
      "cmp %x[M], #0x4\n"
      "bgt 53f\n"
      "beq 40f\n"
      "cmp %x[M], #0x2\n"
      "bgt 27f\n"
      "beq 14f\n"
      "ldr x14, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.s, x20, x13\n"
      "cbz x14, 3f\n"
      "ld1w { z24.s }, p2/Z, [x14]\n"
      "addvl x14, x14, #1\n"
      "b 5f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 4f\n"
      "ld1w { z24.s }, p1/Z, [x11]\n"
      "b 5f\n"
      "4:"  // Height 1: no accumulate
      "mov z24.b, #0x0\n"
      "5:"  // Height 1: setup done
      "mov x10, #0x0\n"
      "6:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 7f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "cbnz x10, 8f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "b 8f\n"
      "7:"  // Height 1: setup direct input
      "mov x28, %x[input_ptr]\n"
      "8:"  // Height 1: input setup done
      "cmp x9, #0x4\n"
      "ble 10f\n"
      "9:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z19.s }, p2/Z, [x12]\n"
      "ld1w { z18.s }, p2/Z, [x12, #1, MUL VL]\n"
      "sub x9, x9, #0x4\n"
      "ld1w { z17.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x12, #3, MUL VL]\n"
      "cmp x9, #0x4\n"
      "addvl x12, x12, #4\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "add x28, x28, #0x10\n"
      "fmla z24.s, z19.s, z0.s[0]\n"
      "fmla z24.s, z18.s, z0.s[1]\n"
      "fmla z24.s, z17.s, z0.s[2]\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "bgt 9b\n"
      "10:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "fmla z24.s, z16.s, z0.s[0]\n"
      "ble 11f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[1]\n"
      "ble 11f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[2]\n"
      "ble 11f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "11:"  // Height 1: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 6b\n"
      "tbz %x[flags], #1, 12f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "fmin z24.s, p2/M, z24.s, z17.s\n"
      "fmax z24.s, p2/M, z24.s, z16.s\n"
      "12:"  // Height 1: No activation
      "st1w { z24.s }, p1, [x11]\n"
      "addvl x11, x11, #1\n"
      "13:"  // Height 1: Writeback done
      "decw x13\n"
      "cmp x13, XZR\n"
      "bgt 2b\n"
      "b 106f\n"
      "14:"  // Height 2
      "ldr x14, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "15:"  // Height 2: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.s, x20, x13\n"
      "cbz x14, 16f\n"
      "ld1w { z24.s }, p2/Z, [x14]\n"
      "addvl x14, x14, #1\n"
      "mov z25.d, z24.d\n"
      "b 18f\n"
      "16:"  // Height 2: no bias
      "tbz %x[flags], #0, 17f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p1/Z, [x11]\n"
      "add x20, x11, x20, LSL #2\n"
      "ld1w { z25.s }, p1/Z, [x20]\n"
      "b 18f\n"
      "17:"  // Height 2: no accumulate
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "18:"  // Height 2: setup done
      "mov x10, #0x0\n"
      "19:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 20f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "cbnz x10, 21f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "b 21f\n"
      "20:"  // Height 2: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "21:"  // Height 2: input setup done
      "cmp x9, #0x4\n"
      "ble 23f\n"
      "22:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z19.s }, p2/Z, [x12]\n"
      "ld1w { z18.s }, p2/Z, [x12, #1, MUL VL]\n"
      "sub x9, x9, #0x4\n"
      "ld1w { z17.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x12, #3, MUL VL]\n"
      "cmp x9, #0x4\n"
      "addvl x12, x12, #4\n"
      "ld1rqw { z1.s }, p0/Z, [x28]\n"
      "add x28, x28, #0x10\n"
      "ld1rqw { z0.s }, p0/Z, [x27]\n"
      "add x27, x27, #0x10\n"
      "fmla z24.s, z19.s, z1.s[0]\n"
      "fmla z25.s, z19.s, z0.s[0]\n"
      "fmla z24.s, z18.s, z1.s[1]\n"
      "fmla z25.s, z18.s, z0.s[1]\n"
      "fmla z24.s, z17.s, z1.s[2]\n"
      "fmla z25.s, z17.s, z0.s[2]\n"
      "fmla z24.s, z16.s, z1.s[3]\n"
      "fmla z25.s, z16.s, z0.s[3]\n"
      "bgt 22b\n"
      "23:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "ld1rqw { z1.s }, p0/Z, [x27]\n"
      "fmla z24.s, z16.s, z0.s[0]\n"
      "fmla z25.s, z16.s, z1.s[0]\n"
      "ble 24f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[1]\n"
      "fmla z25.s, z16.s, z1.s[1]\n"
      "ble 24f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[2]\n"
      "fmla z25.s, z16.s, z1.s[2]\n"
      "ble 24f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "fmla z25.s, z16.s, z1.s[3]\n"
      "24:"  // Height 2: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 19b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "tbz %x[flags], #1, 25f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "fmin z24.s, p2/M, z24.s, z17.s\n"
      "fmin z25.s, p2/M, z25.s, z17.s\n"
      "fmax z24.s, p2/M, z24.s, z16.s\n"
      "fmax z25.s, p2/M, z25.s, z16.s\n"
      "25:"  // Height 2: No activation
      "st1w { z24.s }, p1, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p1, [x28]\n"
      "26:"  // Height 2: Writeback done
      "decw x13\n"
      "cmp x13, XZR\n"
      "bgt 15b\n"
      "b 106f\n"
      "27:"  // Height 3
      "ldr x14, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "28:"  // Height 3: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.s, x20, x13\n"
      "cbz x14, 29f\n"
      "ld1w { z24.s }, p2/Z, [x14]\n"
      "addvl x14, x14, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "b 31f\n"
      "29:"  // Height 3: no bias
      "tbz %x[flags], #0, 30f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p1/Z, [x11]\n"
      "add x20, x11, x21, LSL #2\n"
      "ld1w { z25.s }, p1/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z26.s }, p1/Z, [x20]\n"
      "b 31f\n"
      "30:"  // Height 3: no accumulate
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "31:"  // Height 3: setup done
      "mov x10, #0x0\n"
      "32:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 33f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "cbnz x10, 34f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "b 34f\n"
      "33:"  // Height 3: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "34:"  // Height 3: input setup done
      "cmp x9, #0x4\n"
      "ble 36f\n"
      "35:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z19.s }, p2/Z, [x12]\n"
      "ld1w { z18.s }, p2/Z, [x12, #1, MUL VL]\n"
      "sub x9, x9, #0x4\n"
      "ld1w { z17.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x12, #3, MUL VL]\n"
      "cmp x9, #0x4\n"
      "addvl x12, x12, #4\n"
      "ld1rqw { z2.s }, p0/Z, [x28]\n"
      "add x28, x28, #0x10\n"
      "ld1rqw { z1.s }, p0/Z, [x27]\n"
      "add x27, x27, #0x10\n"
      "ld1rqw { z0.s }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "fmla z24.s, z19.s, z2.s[0]\n"
      "fmla z25.s, z19.s, z1.s[0]\n"
      "fmla z26.s, z19.s, z0.s[0]\n"
      "fmla z24.s, z18.s, z2.s[1]\n"
      "fmla z25.s, z18.s, z1.s[1]\n"
      "fmla z26.s, z18.s, z0.s[1]\n"
      "fmla z24.s, z17.s, z2.s[2]\n"
      "fmla z25.s, z17.s, z1.s[2]\n"
      "fmla z26.s, z17.s, z0.s[2]\n"
      "fmla z24.s, z16.s, z2.s[3]\n"
      "fmla z25.s, z16.s, z1.s[3]\n"
      "fmla z26.s, z16.s, z0.s[3]\n"
      "bgt 35b\n"
      "36:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "ld1rqw { z1.s }, p0/Z, [x27]\n"
      "ld1rqw { z2.s }, p0/Z, [x26]\n"
      "fmla z24.s, z16.s, z0.s[0]\n"
      "fmla z25.s, z16.s, z1.s[0]\n"
      "fmla z26.s, z16.s, z2.s[0]\n"
      "ble 37f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[1]\n"
      "fmla z25.s, z16.s, z1.s[1]\n"
      "fmla z26.s, z16.s, z2.s[1]\n"
      "ble 37f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[2]\n"
      "fmla z25.s, z16.s, z1.s[2]\n"
      "fmla z26.s, z16.s, z2.s[2]\n"
      "ble 37f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "fmla z25.s, z16.s, z1.s[3]\n"
      "fmla z26.s, z16.s, z2.s[3]\n"
      "37:"  // Height 3: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 32b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "tbz %x[flags], #1, 38f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "fmin z24.s, p2/M, z24.s, z17.s\n"
      "fmin z25.s, p2/M, z25.s, z17.s\n"
      "fmin z26.s, p2/M, z26.s, z17.s\n"
      "fmax z24.s, p2/M, z24.s, z16.s\n"
      "fmax z25.s, p2/M, z25.s, z16.s\n"
      "fmax z26.s, p2/M, z26.s, z16.s\n"
      "38:"  // Height 3: No activation
      "st1w { z24.s }, p1, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p1, [x28]\n"
      "st1w { z26.s }, p1, [x27]\n"
      "39:"  // Height 3: Writeback done
      "decw x13\n"
      "cmp x13, XZR\n"
      "bgt 28b\n"
      "b 106f\n"
      "40:"  // Height 4
      "ldr x14, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "41:"  // Height 4: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.s, x20, x13\n"
      "cbz x14, 42f\n"
      "ld1w { z24.s }, p2/Z, [x14]\n"
      "addvl x14, x14, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "b 44f\n"
      "42:"  // Height 4: no bias
      "tbz %x[flags], #0, 43f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p1/Z, [x11]\n"
      "add x20, x11, x21, LSL #2\n"
      "ld1w { z25.s }, p1/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z26.s }, p1/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z27.s }, p1/Z, [x20]\n"
      "b 44f\n"
      "43:"  // Height 4: no accumulate
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "44:"  // Height 4: setup done
      "mov x10, #0x0\n"
      "45:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 46f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "cbnz x10, 47f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "b 47f\n"
      "46:"  // Height 4: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "47:"  // Height 4: input setup done
      "cmp x9, #0x4\n"
      "ble 49f\n"
      "48:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z19.s }, p2/Z, [x12]\n"
      "ld1w { z18.s }, p2/Z, [x12, #1, MUL VL]\n"
      "sub x9, x9, #0x4\n"
      "ld1w { z17.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x12, #3, MUL VL]\n"
      "cmp x9, #0x4\n"
      "addvl x12, x12, #4\n"
      "ld1rqw { z3.s }, p0/Z, [x28]\n"
      "add x28, x28, #0x10\n"
      "ld1rqw { z2.s }, p0/Z, [x27]\n"
      "add x27, x27, #0x10\n"
      "ld1rqw { z1.s }, p0/Z, [x26]\n"
      "ld1rqw { z0.s }, p0/Z, [x25]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "fmla z24.s, z19.s, z3.s[0]\n"
      "fmla z25.s, z19.s, z2.s[0]\n"
      "fmla z26.s, z19.s, z1.s[0]\n"
      "fmla z27.s, z19.s, z0.s[0]\n"
      "fmla z24.s, z18.s, z3.s[1]\n"
      "fmla z25.s, z18.s, z2.s[1]\n"
      "fmla z26.s, z18.s, z1.s[1]\n"
      "fmla z27.s, z18.s, z0.s[1]\n"
      "fmla z24.s, z17.s, z3.s[2]\n"
      "fmla z25.s, z17.s, z2.s[2]\n"
      "fmla z26.s, z17.s, z1.s[2]\n"
      "fmla z27.s, z17.s, z0.s[2]\n"
      "fmla z24.s, z16.s, z3.s[3]\n"
      "fmla z25.s, z16.s, z2.s[3]\n"
      "fmla z26.s, z16.s, z1.s[3]\n"
      "fmla z27.s, z16.s, z0.s[3]\n"
      "bgt 48b\n"
      "49:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "ld1rqw { z1.s }, p0/Z, [x27]\n"
      "ld1rqw { z2.s }, p0/Z, [x26]\n"
      "ld1rqw { z3.s }, p0/Z, [x25]\n"
      "fmla z24.s, z16.s, z0.s[0]\n"
      "fmla z25.s, z16.s, z1.s[0]\n"
      "fmla z26.s, z16.s, z2.s[0]\n"
      "fmla z27.s, z16.s, z3.s[0]\n"
      "ble 50f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[1]\n"
      "fmla z25.s, z16.s, z1.s[1]\n"
      "fmla z26.s, z16.s, z2.s[1]\n"
      "fmla z27.s, z16.s, z3.s[1]\n"
      "ble 50f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[2]\n"
      "fmla z25.s, z16.s, z1.s[2]\n"
      "fmla z26.s, z16.s, z2.s[2]\n"
      "fmla z27.s, z16.s, z3.s[2]\n"
      "ble 50f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "fmla z25.s, z16.s, z1.s[3]\n"
      "fmla z26.s, z16.s, z2.s[3]\n"
      "fmla z27.s, z16.s, z3.s[3]\n"
      "50:"  // Height 4: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 45b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "tbz %x[flags], #1, 51f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "fmin z24.s, p2/M, z24.s, z17.s\n"
      "fmin z25.s, p2/M, z25.s, z17.s\n"
      "fmin z26.s, p2/M, z26.s, z17.s\n"
      "fmin z27.s, p2/M, z27.s, z17.s\n"
      "fmax z24.s, p2/M, z24.s, z16.s\n"
      "fmax z25.s, p2/M, z25.s, z16.s\n"
      "fmax z26.s, p2/M, z26.s, z16.s\n"
      "fmax z27.s, p2/M, z27.s, z16.s\n"
      "51:"  // Height 4: No activation
      "st1w { z24.s }, p1, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p1, [x28]\n"
      "st1w { z26.s }, p1, [x27]\n"
      "st1w { z27.s }, p1, [x26]\n"
      "52:"  // Height 4: Writeback done
      "decw x13\n"
      "cmp x13, XZR\n"
      "bgt 41b\n"
      "b 106f\n"
      "53:"  // Height 5
      "ldr x14, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "54:"  // Height 5: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.s, x20, x13\n"
      "cbz x14, 55f\n"
      "ld1w { z24.s }, p2/Z, [x14]\n"
      "addvl x14, x14, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "mov z28.d, z24.d\n"
      "b 57f\n"
      "55:"  // Height 5: no bias
      "tbz %x[flags], #0, 56f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p1/Z, [x11]\n"
      "add x20, x11, x21, LSL #2\n"
      "ld1w { z25.s }, p1/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z26.s }, p1/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z27.s }, p1/Z, [x20]\n"
      "add x20, x20, x21, LSL #2\n"
      "ld1w { z28.s }, p1/Z, [x20]\n"
      "b 57f\n"
      "56:"  // Height 5: no accumulate
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "57:"  // Height 5: setup done
      "mov x10, #0x0\n"
      "58:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 59f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "cbnz x10, 60f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "b 60f\n"
      "59:"  // Height 5: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "60:"  // Height 5: input setup done
      "cmp x9, #0x4\n"
      "ble 62f\n"
      "61:"  // Height 5: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z19.s }, p2/Z, [x12]\n"
      "ld1w { z18.s }, p2/Z, [x12, #1, MUL VL]\n"
      "sub x9, x9, #0x4\n"
      "ld1w { z17.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x12, #3, MUL VL]\n"
      "cmp x9, #0x4\n"
      "addvl x12, x12, #4\n"
      "ld1rqw { z4.s }, p0/Z, [x28]\n"
      "add x28, x28, #0x10\n"
      "ld1rqw { z3.s }, p0/Z, [x27]\n"
      "add x27, x27, #0x10\n"
      "ld1rqw { z2.s }, p0/Z, [x26]\n"
      "ld1rqw { z1.s }, p0/Z, [x25]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "ld1rqw { z0.s }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "fmla z24.s, z19.s, z4.s[0]\n"
      "fmla z25.s, z19.s, z3.s[0]\n"
      "fmla z26.s, z19.s, z2.s[0]\n"
      "fmla z27.s, z19.s, z1.s[0]\n"
      "fmla z28.s, z19.s, z0.s[0]\n"
      "fmla z24.s, z18.s, z4.s[1]\n"
      "fmla z25.s, z18.s, z3.s[1]\n"
      "fmla z26.s, z18.s, z2.s[1]\n"
      "fmla z27.s, z18.s, z1.s[1]\n"
      "fmla z28.s, z18.s, z0.s[1]\n"
      "fmla z24.s, z17.s, z4.s[2]\n"
      "fmla z25.s, z17.s, z3.s[2]\n"
      "fmla z26.s, z17.s, z2.s[2]\n"
      "fmla z27.s, z17.s, z1.s[2]\n"
      "fmla z28.s, z17.s, z0.s[2]\n"
      "fmla z24.s, z16.s, z4.s[3]\n"
      "fmla z25.s, z16.s, z3.s[3]\n"
      "fmla z26.s, z16.s, z2.s[3]\n"
      "fmla z27.s, z16.s, z1.s[3]\n"
      "fmla z28.s, z16.s, z0.s[3]\n"
      "bgt 61b\n"
      "62:"  // Height 5: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "ld1rqw { z1.s }, p0/Z, [x27]\n"
      "ld1rqw { z2.s }, p0/Z, [x26]\n"
      "ld1rqw { z3.s }, p0/Z, [x25]\n"
      "ld1rqw { z4.s }, p0/Z, [x24]\n"
      "fmla z24.s, z16.s, z0.s[0]\n"
      "fmla z25.s, z16.s, z1.s[0]\n"
      "fmla z26.s, z16.s, z2.s[0]\n"
      "fmla z27.s, z16.s, z3.s[0]\n"
      "fmla z28.s, z16.s, z4.s[0]\n"
      "ble 63f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[1]\n"
      "fmla z25.s, z16.s, z1.s[1]\n"
      "fmla z26.s, z16.s, z2.s[1]\n"
      "fmla z27.s, z16.s, z3.s[1]\n"
      "fmla z28.s, z16.s, z4.s[1]\n"
      "ble 63f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[2]\n"
      "fmla z25.s, z16.s, z1.s[2]\n"
      "fmla z26.s, z16.s, z2.s[2]\n"
      "fmla z27.s, z16.s, z3.s[2]\n"
      "fmla z28.s, z16.s, z4.s[2]\n"
      "ble 63f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "fmla z25.s, z16.s, z1.s[3]\n"
      "fmla z26.s, z16.s, z2.s[3]\n"
      "fmla z27.s, z16.s, z3.s[3]\n"
      "fmla z28.s, z16.s, z4.s[3]\n"
      "63:"  // Height 5: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 58b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "tbz %x[flags], #1, 64f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "fmin z24.s, p2/M, z24.s, z17.s\n"
      "fmin z25.s, p2/M, z25.s, z17.s\n"
      "fmin z26.s, p2/M, z26.s, z17.s\n"
      "fmin z27.s, p2/M, z27.s, z17.s\n"
      "fmin z28.s, p2/M, z28.s, z17.s\n"
      "fmax z24.s, p2/M, z24.s, z16.s\n"
      "fmax z25.s, p2/M, z25.s, z16.s\n"
      "fmax z26.s, p2/M, z26.s, z16.s\n"
      "fmax z27.s, p2/M, z27.s, z16.s\n"
      "fmax z28.s, p2/M, z28.s, z16.s\n"
      "64:"  // Height 5: No activation
      "st1w { z24.s }, p1, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p1, [x28]\n"
      "st1w { z26.s }, p1, [x27]\n"
      "st1w { z27.s }, p1, [x26]\n"
      "st1w { z28.s }, p1, [x25]\n"
      "65:"  // Height 5: Writeback done
      "decw x13\n"
      "cmp x13, XZR\n"
      "bgt 54b\n"
      "b 106f\n"
      "66:"  // Height 6
      "ldr x14, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "67:"  // Height 6: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.s, x20, x13\n"
      "cbz x14, 68f\n"
      "ld1w { z24.s }, p2/Z, [x14]\n"
      "addvl x14, x14, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "mov z28.d, z24.d\n"
      "mov z29.d, z24.d\n"
      "b 70f\n"
      "68:"  // Height 6: no bias
      "tbz %x[flags], #0, 69f\n"
      "ldr x22, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p1/Z, [x11]\n"
      "add x20, x11, x22, LSL #2\n"
      "ld1w { z25.s }, p1/Z, [x20]\n"
      "add x20, x20, x22, LSL #2\n"
      "ld1w { z26.s }, p1/Z, [x20]\n"
      "add x20, x20, x22, LSL #2\n"
      "ld1w { z27.s }, p1/Z, [x20]\n"
      "add x21, x20, x22, LSL #2\n"
      "add x20, x21, x22, LSL #2\n"
      "ld1w { z28.s }, p1/Z, [x21]\n"
      "ld1w { z29.s }, p1/Z, [x20]\n"
      "b 70f\n"
      "69:"  // Height 6: no accumulate
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "mov z29.b, #0x0\n"
      "70:"  // Height 6: setup done
      "mov x10, #0x0\n"
      "71:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 72f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "cbnz x10, 73f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "b 73f\n"
      "72:"  // Height 6: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "73:"  // Height 6: input setup done
      "cmp x9, #0x4\n"
      "ble 75f\n"
      "74:"  // Height 6: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z19.s }, p2/Z, [x12]\n"
      "ld1w { z18.s }, p2/Z, [x12, #1, MUL VL]\n"
      "sub x9, x9, #0x4\n"
      "ld1w { z17.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x12, #3, MUL VL]\n"
      "cmp x9, #0x4\n"
      "addvl x12, x12, #4\n"
      "ld1rqw { z5.s }, p0/Z, [x28]\n"
      "add x28, x28, #0x10\n"
      "ld1rqw { z4.s }, p0/Z, [x27]\n"
      "add x27, x27, #0x10\n"
      "ld1rqw { z3.s }, p0/Z, [x26]\n"
      "ld1rqw { z2.s }, p0/Z, [x25]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "ld1rqw { z1.s }, p0/Z, [x24]\n"
      "ld1rqw { z0.s }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "fmla z24.s, z19.s, z5.s[0]\n"
      "fmla z25.s, z19.s, z4.s[0]\n"
      "fmla z26.s, z19.s, z3.s[0]\n"
      "fmla z27.s, z19.s, z2.s[0]\n"
      "fmla z28.s, z19.s, z1.s[0]\n"
      "fmla z29.s, z19.s, z0.s[0]\n"
      "fmla z24.s, z18.s, z5.s[1]\n"
      "fmla z25.s, z18.s, z4.s[1]\n"
      "fmla z26.s, z18.s, z3.s[1]\n"
      "fmla z27.s, z18.s, z2.s[1]\n"
      "fmla z28.s, z18.s, z1.s[1]\n"
      "fmla z29.s, z18.s, z0.s[1]\n"
      "fmla z24.s, z17.s, z5.s[2]\n"
      "fmla z25.s, z17.s, z4.s[2]\n"
      "fmla z26.s, z17.s, z3.s[2]\n"
      "fmla z27.s, z17.s, z2.s[2]\n"
      "fmla z28.s, z17.s, z1.s[2]\n"
      "fmla z29.s, z17.s, z0.s[2]\n"
      "fmla z24.s, z16.s, z5.s[3]\n"
      "fmla z25.s, z16.s, z4.s[3]\n"
      "fmla z26.s, z16.s, z3.s[3]\n"
      "fmla z27.s, z16.s, z2.s[3]\n"
      "fmla z28.s, z16.s, z1.s[3]\n"
      "fmla z29.s, z16.s, z0.s[3]\n"
      "bgt 74b\n"
      "75:"  // Height 6: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "ld1rqw { z1.s }, p0/Z, [x27]\n"
      "ld1rqw { z2.s }, p0/Z, [x26]\n"
      "ld1rqw { z3.s }, p0/Z, [x25]\n"
      "ld1rqw { z4.s }, p0/Z, [x24]\n"
      "ld1rqw { z5.s }, p0/Z, [x23]\n"
      "fmla z24.s, z16.s, z0.s[0]\n"
      "fmla z25.s, z16.s, z1.s[0]\n"
      "fmla z26.s, z16.s, z2.s[0]\n"
      "fmla z27.s, z16.s, z3.s[0]\n"
      "fmla z28.s, z16.s, z4.s[0]\n"
      "fmla z29.s, z16.s, z5.s[0]\n"
      "ble 76f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[1]\n"
      "fmla z25.s, z16.s, z1.s[1]\n"
      "fmla z26.s, z16.s, z2.s[1]\n"
      "fmla z27.s, z16.s, z3.s[1]\n"
      "fmla z28.s, z16.s, z4.s[1]\n"
      "fmla z29.s, z16.s, z5.s[1]\n"
      "ble 76f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[2]\n"
      "fmla z25.s, z16.s, z1.s[2]\n"
      "fmla z26.s, z16.s, z2.s[2]\n"
      "fmla z27.s, z16.s, z3.s[2]\n"
      "fmla z28.s, z16.s, z4.s[2]\n"
      "fmla z29.s, z16.s, z5.s[2]\n"
      "ble 76f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "fmla z25.s, z16.s, z1.s[3]\n"
      "fmla z26.s, z16.s, z2.s[3]\n"
      "fmla z27.s, z16.s, z3.s[3]\n"
      "fmla z28.s, z16.s, z4.s[3]\n"
      "fmla z29.s, z16.s, z5.s[3]\n"
      "76:"  // Height 6: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 71b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "tbz %x[flags], #1, 77f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "fmin z24.s, p2/M, z24.s, z17.s\n"
      "fmin z25.s, p2/M, z25.s, z17.s\n"
      "fmin z26.s, p2/M, z26.s, z17.s\n"
      "fmin z27.s, p2/M, z27.s, z17.s\n"
      "fmin z28.s, p2/M, z28.s, z17.s\n"
      "fmin z29.s, p2/M, z29.s, z17.s\n"
      "fmax z24.s, p2/M, z24.s, z16.s\n"
      "fmax z25.s, p2/M, z25.s, z16.s\n"
      "fmax z26.s, p2/M, z26.s, z16.s\n"
      "fmax z27.s, p2/M, z27.s, z16.s\n"
      "fmax z28.s, p2/M, z28.s, z16.s\n"
      "fmax z29.s, p2/M, z29.s, z16.s\n"
      "77:"  // Height 6: No activation
      "st1w { z24.s }, p1, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p1, [x28]\n"
      "st1w { z26.s }, p1, [x27]\n"
      "st1w { z27.s }, p1, [x26]\n"
      "st1w { z28.s }, p1, [x25]\n"
      "st1w { z29.s }, p1, [x24]\n"
      "78:"  // Height 6: Writeback done
      "decw x13\n"
      "cmp x13, XZR\n"
      "bgt 67b\n"
      "b 106f\n"
      "79:"  // Height 7
      "ldr x14, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "80:"  // Height 7: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.s, x20, x13\n"
      "cbz x14, 81f\n"
      "ld1w { z24.s }, p2/Z, [x14]\n"
      "addvl x14, x14, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "mov z28.d, z24.d\n"
      "mov z29.d, z24.d\n"
      "mov z30.d, z24.d\n"
      "b 83f\n"
      "81:"  // Height 7: no bias
      "tbz %x[flags], #0, 82f\n"
      "ldr x23, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p1/Z, [x11]\n"
      "add x20, x11, x23, LSL #2\n"
      "ld1w { z25.s }, p1/Z, [x20]\n"
      "add x20, x20, x23, LSL #2\n"
      "ld1w { z26.s }, p1/Z, [x20]\n"
      "add x20, x20, x23, LSL #2\n"
      "ld1w { z27.s }, p1/Z, [x20]\n"
      "add x22, x20, x23, LSL #2\n"
      "add x21, x22, x23, LSL #2\n"
      "add x20, x21, x23, LSL #2\n"
      "ld1w { z28.s }, p1/Z, [x22]\n"
      "ld1w { z29.s }, p1/Z, [x21]\n"
      "ld1w { z30.s }, p1/Z, [x20]\n"
      "b 83f\n"
      "82:"  // Height 7: no accumulate
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "mov z29.b, #0x0\n"
      "mov z30.b, #0x0\n"
      "83:"  // Height 7: setup done
      "mov x10, #0x0\n"
      "84:"  // Height 7: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 85f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "ldr x22, [x20, #0x30]\n"
      "cbnz x10, 86f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "b 86f\n"
      "85:"  // Height 7: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "86:"  // Height 7: input setup done
      "cmp x9, #0x4\n"
      "ble 88f\n"
      "87:"  // Height 7: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z19.s }, p2/Z, [x12]\n"
      "ld1w { z18.s }, p2/Z, [x12, #1, MUL VL]\n"
      "sub x9, x9, #0x4\n"
      "ld1w { z17.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x12, #3, MUL VL]\n"
      "cmp x9, #0x4\n"
      "addvl x12, x12, #4\n"
      "ld1rqw { z6.s }, p0/Z, [x28]\n"
      "add x28, x28, #0x10\n"
      "ld1rqw { z5.s }, p0/Z, [x27]\n"
      "add x27, x27, #0x10\n"
      "ld1rqw { z4.s }, p0/Z, [x26]\n"
      "ld1rqw { z3.s }, p0/Z, [x25]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "ld1rqw { z2.s }, p0/Z, [x24]\n"
      "ld1rqw { z1.s }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "ld1rqw { z0.s }, p0/Z, [x22]\n"
      "fmla z24.s, z19.s, z6.s[0]\n"
      "fmla z25.s, z19.s, z5.s[0]\n"
      "add x22, x22, #0x10\n"
      "fmla z26.s, z19.s, z4.s[0]\n"
      "fmla z27.s, z19.s, z3.s[0]\n"
      "fmla z28.s, z19.s, z2.s[0]\n"
      "fmla z29.s, z19.s, z1.s[0]\n"
      "fmla z30.s, z19.s, z0.s[0]\n"
      "fmla z24.s, z18.s, z6.s[1]\n"
      "fmla z25.s, z18.s, z5.s[1]\n"
      "fmla z26.s, z18.s, z4.s[1]\n"
      "fmla z27.s, z18.s, z3.s[1]\n"
      "fmla z28.s, z18.s, z2.s[1]\n"
      "fmla z29.s, z18.s, z1.s[1]\n"
      "fmla z30.s, z18.s, z0.s[1]\n"
      "fmla z24.s, z17.s, z6.s[2]\n"
      "fmla z25.s, z17.s, z5.s[2]\n"
      "fmla z26.s, z17.s, z4.s[2]\n"
      "fmla z27.s, z17.s, z3.s[2]\n"
      "fmla z28.s, z17.s, z2.s[2]\n"
      "fmla z29.s, z17.s, z1.s[2]\n"
      "fmla z30.s, z17.s, z0.s[2]\n"
      "fmla z24.s, z16.s, z6.s[3]\n"
      "fmla z25.s, z16.s, z5.s[3]\n"
      "fmla z26.s, z16.s, z4.s[3]\n"
      "fmla z27.s, z16.s, z3.s[3]\n"
      "fmla z28.s, z16.s, z2.s[3]\n"
      "fmla z29.s, z16.s, z1.s[3]\n"
      "fmla z30.s, z16.s, z0.s[3]\n"
      "bgt 87b\n"
      "88:"  // Height 7: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "ld1rqw { z1.s }, p0/Z, [x27]\n"
      "ld1rqw { z2.s }, p0/Z, [x26]\n"
      "ld1rqw { z3.s }, p0/Z, [x25]\n"
      "ld1rqw { z4.s }, p0/Z, [x24]\n"
      "ld1rqw { z5.s }, p0/Z, [x23]\n"
      "ld1rqw { z6.s }, p0/Z, [x22]\n"
      "fmla z24.s, z16.s, z0.s[0]\n"
      "fmla z25.s, z16.s, z1.s[0]\n"
      "fmla z26.s, z16.s, z2.s[0]\n"
      "fmla z27.s, z16.s, z3.s[0]\n"
      "fmla z28.s, z16.s, z4.s[0]\n"
      "fmla z29.s, z16.s, z5.s[0]\n"
      "fmla z30.s, z16.s, z6.s[0]\n"
      "ble 89f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[1]\n"
      "fmla z25.s, z16.s, z1.s[1]\n"
      "fmla z26.s, z16.s, z2.s[1]\n"
      "fmla z27.s, z16.s, z3.s[1]\n"
      "fmla z28.s, z16.s, z4.s[1]\n"
      "fmla z29.s, z16.s, z5.s[1]\n"
      "fmla z30.s, z16.s, z6.s[1]\n"
      "ble 89f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[2]\n"
      "fmla z25.s, z16.s, z1.s[2]\n"
      "fmla z26.s, z16.s, z2.s[2]\n"
      "fmla z27.s, z16.s, z3.s[2]\n"
      "fmla z28.s, z16.s, z4.s[2]\n"
      "fmla z29.s, z16.s, z5.s[2]\n"
      "fmla z30.s, z16.s, z6.s[2]\n"
      "ble 89f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "fmla z25.s, z16.s, z1.s[3]\n"
      "fmla z26.s, z16.s, z2.s[3]\n"
      "fmla z27.s, z16.s, z3.s[3]\n"
      "fmla z28.s, z16.s, z4.s[3]\n"
      "fmla z29.s, z16.s, z5.s[3]\n"
      "fmla z30.s, z16.s, z6.s[3]\n"
      "89:"  // Height 7: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 84b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "tbz %x[flags], #1, 90f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "fmin z24.s, p2/M, z24.s, z17.s\n"
      "fmin z25.s, p2/M, z25.s, z17.s\n"
      "fmin z26.s, p2/M, z26.s, z17.s\n"
      "fmin z27.s, p2/M, z27.s, z17.s\n"
      "fmin z28.s, p2/M, z28.s, z17.s\n"
      "fmin z29.s, p2/M, z29.s, z17.s\n"
      "fmin z30.s, p2/M, z30.s, z17.s\n"
      "fmax z24.s, p2/M, z24.s, z16.s\n"
      "fmax z25.s, p2/M, z25.s, z16.s\n"
      "fmax z26.s, p2/M, z26.s, z16.s\n"
      "fmax z27.s, p2/M, z27.s, z16.s\n"
      "fmax z28.s, p2/M, z28.s, z16.s\n"
      "fmax z29.s, p2/M, z29.s, z16.s\n"
      "fmax z30.s, p2/M, z30.s, z16.s\n"
      "90:"  // Height 7: No activation
      "st1w { z24.s }, p1, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p1, [x28]\n"
      "st1w { z26.s }, p1, [x27]\n"
      "st1w { z27.s }, p1, [x26]\n"
      "st1w { z28.s }, p1, [x25]\n"
      "st1w { z29.s }, p1, [x24]\n"
      "st1w { z30.s }, p1, [x23]\n"
      "91:"  // Height 7: Writeback done
      "decw x13\n"
      "cmp x13, XZR\n"
      "bgt 80b\n"
      "b 106f\n"
      "92:"  // Height 8
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x20\n"
      "ldr x14, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x11\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "93:"  // Height 8: Column loop
      "mov x20, #0x0\n"
      "whilelt p1.s, x20, x13\n"
      "cbz x14, 94f\n"
      "ld1w { z24.s }, p2/Z, [x14]\n"
      "addvl x14, x14, #1\n"
      "mov z25.d, z24.d\n"
      "mov z26.d, z24.d\n"
      "mov z27.d, z24.d\n"
      "mov z28.d, z24.d\n"
      "mov z29.d, z24.d\n"
      "mov z30.d, z24.d\n"
      "mov z31.d, z24.d\n"
      "b 96f\n"
      "94:"  // Height 8: no bias
      "tbz %x[flags], #0, 95f\n"
      "ldr x23, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z24.s }, p1/Z, [x11]\n"
      "add x20, x11, x23, LSL #2\n"
      "ld1w { z25.s }, p1/Z, [x20]\n"
      "add x20, x20, x23, LSL #2\n"
      "ld1w { z26.s }, p1/Z, [x20]\n"
      "add x20, x20, x23, LSL #2\n"
      "ld1w { z27.s }, p1/Z, [x20]\n"
      "add x22, x20, x23, LSL #2\n"
      "add x20, x22, x23, LSL #2\n"
      "add x21, x20, x23, LSL #2\n"
      "ld1w { z28.s }, p1/Z, [x22]\n"
      "ld1w { z29.s }, p1/Z, [x20]\n"
      "add x20, x21, x23, LSL #2\n"
      "ld1w { z30.s }, p1/Z, [x21]\n"
      "ld1w { z31.s }, p1/Z, [x20]\n"
      "b 96f\n"
      "95:"  // Height 8: no accumulate
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "mov z29.b, #0x0\n"
      "mov z30.b, #0x0\n"
      "mov z31.b, #0x0\n"
      "96:"  // Height 8: setup done
      "mov x10, #0x0\n"
      "97:"  // Height 8: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w9, [x20, x10, LSL #0x2]\n"
      "tbz %x[flags], #3, 98f\n"
      "ldr x20, [%x[input_ptr], x10, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x28, [x20, #0x0]\n"
      "ldr x27, [x20, #0x8]\n"
      "ldr x26, [x20, #0x10]\n"
      "ldr x25, [x20, #0x18]\n"
      "ldr x24, [x20, #0x20]\n"
      "ldr x23, [x20, #0x28]\n"
      "ldr x22, [x20, #0x30]\n"
      "ldr x21, [x20, #0x38]\n"
      "cbnz x10, 99f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x28, x28, x20, LSL #2\n"
      "add x27, x27, x20, LSL #2\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "add x21, x21, x20, LSL #2\n"
      "b 99f\n"
      "98:"  // Height 8: setup direct input
      "mov x28, %x[input_ptr]\n"
      "add x27, x28, x21, LSL #2\n"
      "add x26, x27, x21, LSL #2\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "add x21, x22, x21, LSL #2\n"
      "99:"  // Height 8: input setup done
      "cmp x9, #0x4\n"
      "ble 101f\n"
      "100:"  // Height 8: Multiply loop: Main loop head
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z19.s }, p2/Z, [x12]\n"
      "ld1w { z18.s }, p2/Z, [x12, #1, MUL VL]\n"
      "sub x9, x9, #0x4\n"
      "ld1w { z17.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x12, #3, MUL VL]\n"
      "cmp x9, #0x4\n"
      "addvl x12, x12, #4\n"
      "ld1rqw { z7.s }, p0/Z, [x28]\n"
      "add x28, x28, #0x10\n"
      "ld1rqw { z6.s }, p0/Z, [x27]\n"
      "add x27, x27, #0x10\n"
      "ld1rqw { z5.s }, p0/Z, [x26]\n"
      "ld1rqw { z4.s }, p0/Z, [x25]\n"
      "add x26, x26, #0x10\n"
      "add x25, x25, #0x10\n"
      "ld1rqw { z3.s }, p0/Z, [x24]\n"
      "ld1rqw { z2.s }, p0/Z, [x23]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      "ld1rqw { z1.s }, p0/Z, [x22]\n"
      "ld1rqw { z0.s }, p0/Z, [x21]\n"
      "fmla z24.s, z19.s, z7.s[0]\n"
      "fmla z25.s, z19.s, z6.s[0]\n"
      "fmla z26.s, z19.s, z5.s[0]\n"
      "fmla z27.s, z19.s, z4.s[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      "fmla z28.s, z19.s, z3.s[0]\n"
      "fmla z29.s, z19.s, z2.s[0]\n"
      "fmla z30.s, z19.s, z1.s[0]\n"
      "fmla z31.s, z19.s, z0.s[0]\n"
      "fmla z24.s, z18.s, z7.s[1]\n"
      "fmla z25.s, z18.s, z6.s[1]\n"
      "fmla z26.s, z18.s, z5.s[1]\n"
      "fmla z27.s, z18.s, z4.s[1]\n"
      "fmla z28.s, z18.s, z3.s[1]\n"
      "fmla z29.s, z18.s, z2.s[1]\n"
      "fmla z30.s, z18.s, z1.s[1]\n"
      "fmla z31.s, z18.s, z0.s[1]\n"
      "fmla z24.s, z17.s, z7.s[2]\n"
      "fmla z25.s, z17.s, z6.s[2]\n"
      "fmla z26.s, z17.s, z5.s[2]\n"
      "fmla z27.s, z17.s, z4.s[2]\n"
      "fmla z28.s, z17.s, z3.s[2]\n"
      "fmla z29.s, z17.s, z2.s[2]\n"
      "fmla z30.s, z17.s, z1.s[2]\n"
      "fmla z31.s, z17.s, z0.s[2]\n"
      "fmla z24.s, z16.s, z7.s[3]\n"
      "fmla z25.s, z16.s, z6.s[3]\n"
      "fmla z26.s, z16.s, z5.s[3]\n"
      "fmla z27.s, z16.s, z4.s[3]\n"
      "fmla z28.s, z16.s, z3.s[3]\n"
      "fmla z29.s, z16.s, z2.s[3]\n"
      "fmla z30.s, z16.s, z1.s[3]\n"
      "fmla z31.s, z16.s, z0.s[3]\n"
      "bgt 100b\n"
      "101:"  // Height 8: Multiply loop: Single iteration only
      "whilelt p0.s, XZR, x9\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "ld1rqw { z0.s }, p0/Z, [x28]\n"
      "ld1rqw { z1.s }, p0/Z, [x27]\n"
      "ld1rqw { z2.s }, p0/Z, [x26]\n"
      "ld1rqw { z3.s }, p0/Z, [x25]\n"
      "ld1rqw { z4.s }, p0/Z, [x24]\n"
      "ld1rqw { z5.s }, p0/Z, [x23]\n"
      "ld1rqw { z6.s }, p0/Z, [x22]\n"
      "ld1rqw { z7.s }, p0/Z, [x21]\n"
      "fmla z24.s, z16.s, z0.s[0]\n"
      "fmla z25.s, z16.s, z1.s[0]\n"
      "fmla z26.s, z16.s, z2.s[0]\n"
      "fmla z27.s, z16.s, z3.s[0]\n"
      "fmla z28.s, z16.s, z4.s[0]\n"
      "fmla z29.s, z16.s, z5.s[0]\n"
      "fmla z30.s, z16.s, z6.s[0]\n"
      "fmla z31.s, z16.s, z7.s[0]\n"
      "ble 102f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[1]\n"
      "fmla z25.s, z16.s, z1.s[1]\n"
      "fmla z26.s, z16.s, z2.s[1]\n"
      "fmla z27.s, z16.s, z3.s[1]\n"
      "fmla z28.s, z16.s, z4.s[1]\n"
      "fmla z29.s, z16.s, z5.s[1]\n"
      "fmla z30.s, z16.s, z6.s[1]\n"
      "fmla z31.s, z16.s, z7.s[1]\n"
      "ble 102f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "subs x9, x9, #0x1\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[2]\n"
      "fmla z25.s, z16.s, z1.s[2]\n"
      "fmla z26.s, z16.s, z2.s[2]\n"
      "fmla z27.s, z16.s, z3.s[2]\n"
      "fmla z28.s, z16.s, z4.s[2]\n"
      "fmla z29.s, z16.s, z5.s[2]\n"
      "fmla z30.s, z16.s, z6.s[2]\n"
      "fmla z31.s, z16.s, z7.s[2]\n"
      "ble 102f\n"
      "ld1w { z16.s }, p2/Z, [x12]\n"
      "addvl x12, x12, #1\n"
      "fmla z24.s, z16.s, z0.s[3]\n"
      "fmla z25.s, z16.s, z1.s[3]\n"
      "fmla z26.s, z16.s, z2.s[3]\n"
      "fmla z27.s, z16.s, z3.s[3]\n"
      "fmla z28.s, z16.s, z4.s[3]\n"
      "fmla z29.s, z16.s, z5.s[3]\n"
      "fmla z30.s, z16.s, z6.s[3]\n"
      "fmla z31.s, z16.s, z7.s[3]\n"
      "102:"  // Height 8: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x10, x10, #0x1\n"
      "cmp x10, x20\n"
      "bne 97b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x28, x11, x20, LSL #2\n"
      "add x27, x28, x20, LSL #2\n"
      "add x26, x27, x20, LSL #2\n"
      "add x25, x26, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "tbz %x[flags], #1, 103f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z17.s }, p2/Z, [x21]\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "fmin z24.s, p2/M, z24.s, z17.s\n"
      "fmin z25.s, p2/M, z25.s, z17.s\n"
      "fmin z26.s, p2/M, z26.s, z17.s\n"
      "fmin z27.s, p2/M, z27.s, z17.s\n"
      "fmin z28.s, p2/M, z28.s, z17.s\n"
      "fmin z29.s, p2/M, z29.s, z17.s\n"
      "fmin z30.s, p2/M, z30.s, z17.s\n"
      "fmin z31.s, p2/M, z31.s, z17.s\n"
      "fmax z24.s, p2/M, z24.s, z16.s\n"
      "fmax z25.s, p2/M, z25.s, z16.s\n"
      "fmax z26.s, p2/M, z26.s, z16.s\n"
      "fmax z27.s, p2/M, z27.s, z16.s\n"
      "fmax z28.s, p2/M, z28.s, z16.s\n"
      "fmax z29.s, p2/M, z29.s, z16.s\n"
      "fmax z30.s, p2/M, z30.s, z16.s\n"
      "fmax z31.s, p2/M, z31.s, z16.s\n"
      "103:"  // Height 8: No activation
      "st1w { z24.s }, p1, [x11]\n"
      "addvl x11, x11, #1\n"
      "st1w { z25.s }, p1, [x28]\n"
      "st1w { z26.s }, p1, [x27]\n"
      "st1w { z27.s }, p1, [x26]\n"
      "st1w { z28.s }, p1, [x25]\n"
      "st1w { z29.s }, p1, [x24]\n"
      "st1w { z30.s }, p1, [x23]\n"
      "st1w { z31.s }, p1, [x22]\n"
      "104:"  // Height 8: Writeback done
      "decw x13\n"
      "cmp x13, XZR\n"
      "bgt 93b\n"
      "subs %x[M], %x[M], #0x8\n"
      "beq 106f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 105f\n"
      "add x21, x21, #0x8\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "105:"  // Update direct input
      "mov x20, #0x20\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "106:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "p2", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z16", "z17", "z18", "z19", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
