/*
 * Classes handling tag patches
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tests/test-utils.h>
#include <tagcoll/patch.h>
#include <tagcoll/coll/simple.h>

namespace std {

template<typename TAG, typename _Traits>
basic_ostream<char, _Traits>& operator<<(basic_ostream<char, _Traits>& out, const std::set<TAG>& tags)
{
	for (typename std::set<TAG>::const_iterator i = tags.begin();
			i != tags.end(); ++i)
		if (i == tags.begin())
			out << *i;
		else
			out << ", " << *i;
	return out;
}

template<typename ITEM, typename TAG>
ostream& operator<<(ostream& out, const tagcoll::Patch<ITEM, TAG>& p)
{
	out << p.item << ": ";
	bool first = true;
	for (typename std::set<TAG>::const_iterator i = p.added.begin();
			i != p.added.end(); ++i)
	{
		if (first)
			first = false;
		else
			out << ", ";
		out << "+" << *i;
	}
	for (typename std::set<TAG>::const_iterator i = p.removed.begin();
			i != p.removed.end(); ++i)
	{
		if (first)
			first = false;
		else
			out << ", ";
		out << "-" << *i;
	}
	return out;

}

}

namespace tut {
using namespace std;
using namespace tagcoll;
using namespace tagcoll::tests;
using namespace wibble::operators;

struct tagcoll_patches_shar {
};
TESTGRP(tagcoll_patches);


// Test empty patches
template<> template<>
void to::test<1>()
{
	Patch<string, int> a("foo");
	ensure_equals(a.item, "foo");
	ensure(a.added.empty());
	ensure(a.removed.empty());

	Patch<string, int> b("foo", std::set<int>(), std::set<int>());
	ensure_equals(b.item, "foo");
	ensure(b.added.empty());
	ensure(b.removed.empty());

	Patch<string, int> c("foo", wibble::Empty<int>(), wibble::Empty<int>());
	ensure_equals(c.item, "foo");
	ensure(c.added.empty());
	ensure(c.removed.empty());

	ensure_equals(a, b);
	ensure_equals(a, c);
	ensure_equals(b, c);
}

// Test the different constructors
template<> template<>
void to::test<2>()
{
	Patch<string, int> a("foo");
	a.add(1);
	a.remove(2);
	ensure_equals(a.item, "foo");
	ensure_equals(a.added.size(), 1u);
	ensure_equals(a.removed.size(), 1u);
	ensure_equals(*a.added.begin(), 1);
	ensure_equals(*a.removed.begin(), 2);

	std::set<int> added;
	std::set<int> removed;
	added.insert(1);
	removed.insert(2);
	Patch<string, int> b("foo", added, removed);
	ensure_equals(b.item, "foo");
	ensure_equals(b.added.size(), 1u);
	ensure_equals(b.removed.size(), 1u);
	ensure_equals(*b.added.begin(), 1);
	ensure_equals(*b.removed.begin(), 2);

	Patch<string, int> c("foo", wibble::singleton(1), wibble::singleton(2));
	ensure_equals(c.item, "foo");
	ensure_equals(c.added.size(), 1u);
	ensure_equals(c.removed.size(), 1u);
	ensure_equals(*c.added.begin(), 1);
	ensure_equals(*c.removed.begin(), 2);

	ensure_equals(a, b);
	ensure_equals(a, c);
	ensure_equals(b, c);
}

// Test non-empty patches
template<> template<>
void to::test<3>()
{
	Patch<string, int> p("foo");
	p.add(1);
	p.add(2);
	p.remove(3);
	p.remove(4);

	// Check that getReverse() actually returns the reverse patch
	Patch<string, int> rp = p.getReverse();
	ensure_equals(p.added, rp.removed);
	ensure_equals(p.removed, rp.added);

	// Check removeRedundant()
	std::set<int> ts;
	ts.insert(1);
	ts.insert(3);
	p.removeRedundant(ts);
	ensure_not_contains(p.added, 1);
	ensure_contains(p.added, 2);
	ensure_contains(p.removed, 3);
	ensure_not_contains(p.removed, 4);
}

template<> template<>
void to::test<4>()
{
#if 0
	string input_coll(
			"a: b, c\n"
			"b:\n"
			"c: \n"
			"d:  c::D, e::F,    f::g\n"
			);
	string output_coll(
			"a: b\n"
			"b: b, c\n"
			"c: \n"
			"d: c::D, c::d, e::F\n"
			`);
	coll::Simple<string, string> result;
#endif
	PatchList<string, string> patches;

	std::set<string> added;
	std::set<string> removed;

	added.insert("b");
	removed.insert("c"); removed.insert("d");
	patches.addPatch(Patch<string, string>("a", added, removed));

	added.clear(); added.insert("b"), added.insert("c"), added.insert("b");
	removed.clear(); removed.insert("a");
	patches.addPatch(Patch<string, string>("b", added, removed));

	added.clear(); added.insert("c::D"), added.insert("c::d");
	removed.clear(); removed.insert("f::g");
	patches.addPatch(Patch<string, string>("d", added, removed));

#if 0
	parseCollection(input_coll, patcher(patches, inserter(result)));

	coll::Simple<string, string> reference;
	parseCollection(output_coll, inserter(reference));

	ensure_coll_equals(reference, result);
#endif
}

// Check addPatchInverted
template<> template<>
void to::test<5>()
{
	PatchList<string, string> patches;
	patches.addPatchInverted(Patch<string, string>(
				"pizza",
				wibble::singleton(string("tomato")),
				wibble::singleton(string("ketchup"))));

	ensure_equals(patches.size(), 2u);

	PatchList<string, string>::const_iterator i = patches.begin();

	ensure_equals(i->first, "ketchup");
	ensure_equals(i->second.added.size(), 0u);
	ensure_equals(i->second.removed.size(), 1u);
	ensure_equals(*i->second.removed.begin(), "pizza");

	++i;

	ensure_equals(i->first, "tomato");
	ensure_equals(i->second.added.size(), 1u);
	ensure_equals(i->second.removed.size(), 0u);
	ensure_equals(*i->second.added.begin(), "pizza");
}

// Test diffing collections
template<> template<>
void to::test<6>()
{
	string in_coll1(
			"a: b, c\n"
			"b: a\n"
			);
	string in_coll2(
			"a: d, c\n"
			"c: a\n"
			);
	coll::Simple<string, string> coll1;
	coll::Simple<string, string> coll2;
	parseCollection(in_coll1, inserter(coll1));
	parseCollection(in_coll2, inserter(coll2));

	PatchList<string, string> patches;
	patches.addPatch(coll1, coll2);

	// Replacing items should work
	PatchList<string, string>::const_iterator i = patches.begin();
	ensure(i != patches.end());
	ensure_equals(i->first, string("a"));
	ensure_equals(i->second.added.size(), 1u);
	ensure_equals(*i->second.added.begin(), string("d"));
	ensure_equals(i->second.removed.size(), 1u);
	ensure_equals(*i->second.removed.begin(), string("b"));

	// Removing items should work
	++i;
	ensure(i != patches.end());
	ensure_equals(i->first, string("b"));
	ensure(i->second.added.empty());
	ensure_equals(i->second.removed.size(), 1u);
	ensure_equals(*i->second.removed.begin(), string("a"));

	// Adding items should work
	++i;
	ensure(i != patches.end());
	ensure_equals(i->first, string("c"));
	ensure(i->second.removed.empty());
	ensure_equals(i->second.added.size(), 1u);
	ensure_equals(*i->second.added.begin(), string("a"));

	++i;
	ensure(i == patches.end());
}


}

#include <tagcoll/TextFormat.tcc>
#include <tagcoll/patch.tcc>
#include <tagcoll/coll/simple.tcc>

// vim:set ts=4 sw=4:
